//
// C++ Implementation: mediadevicemanager
//
// Description: Controls device/medium object handling, providing
//              helper functions for other objects
//
//
// Author: Jeff Mitchell <kde-dev@emailgoeshere.com>, (C) 2006
//
// Copyright: See COPYING file that comes with this distribution
//
//

#include "amarok.h"
#include "amarokconfig.h"
#include "debug.h"
#include "devicemanager.h"
#include "mediadevicemanager.h"
#include "medium.h"

#include <tqptrlist.h>
#include <tqtimer.h>

#include <dcopclient.h>
#include <dcopobject.h>
#include <tdeapplication.h>

typedef Medium::List MediumList;

MediaDeviceManager* MediaDeviceManager::instance()
{
    static MediaDeviceManager dw;
    return &dw;
}

MediaDeviceManager::MediaDeviceManager()
{
    DEBUG_BLOCK
    connect( DeviceManager::instance(), TQT_SIGNAL( mediumAdded( const Medium*, TQString ) ), TQT_SLOT( slotMediumAdded( const Medium*, TQString ) ) );
    connect( DeviceManager::instance(), TQT_SIGNAL( mediumChanged( const Medium*, TQString ) ), TQT_SLOT( slotMediumChanged( const Medium*, TQString ) ) );
    connect( DeviceManager::instance(), TQT_SIGNAL( mediumRemoved( const Medium*, TQString ) ), TQT_SLOT( slotMediumRemoved( const Medium*, TQString ) ) );
    Medium::List mediums = DeviceManager::instance()->getDeviceList();
    foreachType( Medium::List, mediums )
    {
        slotMediumAdded( &(*it), (*it).id() );
    }
    if( !mediums.count() )
    {
        debug() << "DeviceManager didn't return any devices, we are probably running on a non-KDE system. Trying to reinit media devices later" << endl;
        TQTimer::singleShot( 4000, this, TQT_SLOT( reinitDevices() ) );
    }
    //load manual devices
    TQStringList manualDevices;
    TDEConfig *config = Amarok::config( "MediaBrowser" );
    TQMap<TQString,TQString> savedDevices = config->entryMap( "MediaBrowser" );
    TQMap<TQString,TQString>::Iterator qit;
    TQString curr, currMountPoint, currName;
    for( qit = savedDevices.begin(); qit != savedDevices.end(); ++qit )
    {
        //only handle manual devices, autodetected devices should be added on the fly
        if( qit.key().startsWith( "manual|" ) )
        {
            curr = qit.key();
            curr = curr.remove( "manual|" );
            currName = curr.left( curr.find( '|' ) );
            currMountPoint = curr.remove( currName + '|' );
            manualDevices.append( "false" );           //autodetected
            manualDevices.append( qit.key() );          //id
            manualDevices.append( currName );          //name
            manualDevices.append( currName );          //label
            manualDevices.append( TQString() );     //userLabel
            manualDevices.append( "unknown" );         //mountable?
            manualDevices.append( TQString() );     //device node
            manualDevices.append( currMountPoint );    //mountPoint
            manualDevices.append( "manual" );          //fsType
            manualDevices.append( "unknown" );         //mounted
            manualDevices.append( TQString() );     //baseURL
            manualDevices.append( TQString() );     //MIMEtype
            manualDevices.append( TQString() );     //iconName
            manualDevices.append( "false" );           //encrypted
            manualDevices.append( TQString() );     //clearDeviceUdi
            manualDevices.append( "---" );             //separator
        }
    }
    Medium::List manualMediums = Medium::createList( manualDevices );
    foreachType( Medium::List, manualMediums )
    {
        slotMediumAdded( &(*it), (*it).id() );
    }
}

MediaDeviceManager::~MediaDeviceManager()
{
}

void
MediaDeviceManager::addManualDevice( Medium* added )
{
    m_mediumMap[added->name()] = added;
    added->setFsType( "manual" );
    emit mediumAdded( added, added->name() );
}

void
MediaDeviceManager::removeManualDevice( Medium* removed )
{
    emit mediumRemoved( removed, removed->name() );
    if( m_mediumMap.contains( removed->name() ) )
        m_mediumMap.remove( removed->name() );
}

void MediaDeviceManager::slotMediumAdded( const Medium *m, TQString id)
{
    DEBUG_BLOCK
    if ( m )
    {
        if ( m->fsType() == "manual" ||
                ( !m->deviceNode().startsWith( "/dev/hd" ) &&
                  (m->fsType() == "vfat" || m->fsType() == "hfsplus" || m->fsType() == "msdosfs" ) ) )
            // add other fsTypes that should be auto-detected here later
        {
            if ( m_mediumMap.contains( m->name() ) )
            {
                Medium *tempMedium = m_mediumMap[m->name()];
                m_mediumMap.remove( m->name() );
                delete tempMedium;
            }
            m_mediumMap[m->name()] = new Medium( m );
            emit mediumAdded( m, id );
        }
    }
}

void MediaDeviceManager::slotMediumChanged( const Medium *m, TQString id )
{
    //nothing to do here
    emit mediumChanged( m, id);
}

void MediaDeviceManager::slotMediumRemoved( const Medium* , TQString id )
{
    DEBUG_BLOCK
    Medium* removedMedium = 0;
    if ( m_mediumMap.contains(id) )
        removedMedium = m_mediumMap[id];
    if ( removedMedium )
        debug() << "[MediaDeviceManager::slotMediumRemoved] Obtained medium name is " << id << ", id is: " << removedMedium->id() << endl;
    else
        debug() << "[MediaDeviceManager::slotMediumRemoved] Medium was unknown and is null; name was " << id << endl;
    //if you get a null pointer from this signal, it means we did not know about the device
    //before it was removed, i.e. the removal was the first event for the device received while amarok
    //has been running
    //There is no point in calling getDevice, since it will not be in the list anyways
    emit mediumRemoved( removedMedium, id );
    if ( m_mediumMap.contains(id) )
        m_mediumMap.remove(id);
    delete removedMedium;
}

Medium* MediaDeviceManager::getDevice( TQString name )
{
    return DeviceManager::instance()->getDevice( name );
}

void MediaDeviceManager::reinitDevices( )
{
    Medium::List mediums = DeviceManager::instance()->getDeviceList();
    foreachType( Medium::List, mediums )
    {
        slotMediumAdded( &(*it), (*it).id() );
    }
}

#include "mediadevicemanager.moc"
