/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 *
 * Date        : 2006-22-01
 * Description : digikamalbum TDEIO slave interface to get image 
 *               info from database.
 *
 * Copyright (C) 2007 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * ============================================================ */

// TQt includes.

#include <tqstring.h>
#include <tqdatastream.h>

// KDE includes.

#include <tdeio/job.h>
#include <kurl.h>

// Local includes.

#include "ddebug.h"
#include "album.h"
#include "albummanager.h"
#include "albumsettings.h"
#include "imageinfojob.h"
#include "imageinfojob.moc"

namespace Digikam
{

class ImageInfoJobPriv
{
public:

    ImageInfoJobPriv()
    {
        job         = 0;

        AlbumSettings *settings = AlbumSettings::instance();
        imagefilter = settings->getImageFileFilter().lower() +
                      settings->getImageFileFilter().upper() +
                      settings->getRawFileFilter().lower()   +
                      settings->getRawFileFilter().upper();
    }

    TQString            imagefilter; 

    TDEIO::TransferJob  *job;
};

ImageInfoJob::ImageInfoJob()
{
    d = new ImageInfoJobPriv;
}

ImageInfoJob::~ImageInfoJob()
{
    delete d;
}

void ImageInfoJob::allItemsFromAlbum(Album *album)
{
    if (d->job)
    {
        d->job->kill();
        d->job = 0;
    }

    if (!album)
        return;
    
    TQByteArray  ba;
    TQDataStream ds(ba, IO_WriteOnly);
    ds << AlbumManager::instance()->getLibraryPath();
    ds << album->kurl();
    ds << d->imagefilter;
    ds << 0; // getting dimensions (not needed here)
    ds << 0; // recursive sub-album (not needed here)
    ds << 0; // recursive sub-tags (not needed here)

    // Protocol = digikamalbums -> tdeio_digikamalbums
    d->job = new TDEIO::TransferJob(album->kurl(), TDEIO::CMD_SPECIAL,
                                  ba, TQByteArray(), false);

    connect(d->job, TQT_SIGNAL(result(TDEIO::Job*)),
            this, TQT_SLOT(slotResult(TDEIO::Job*)));

    connect(d->job, TQT_SIGNAL(data(TDEIO::Job*, const TQByteArray&)),
            this, TQT_SLOT(slotData(TDEIO::Job*, const TQByteArray&)));
}

void ImageInfoJob::stop()
{
    if (d->job)
    {
        d->job->kill();
        d->job = 0;
    }
}

void ImageInfoJob::slotResult(TDEIO::Job* job)
{
    d->job = 0;

    if (job->error())
    {
        DWarning() << "Failed to list url: " << job->errorString() << endl;
        return;
    }

    emit signalCompleted();
}

void ImageInfoJob::slotData(TDEIO::Job*, const TQByteArray& data)
{
    if (data.isEmpty())
        return;

    TQ_LLONG       imageID;
    int           albumID;
    TQString       name;
    TQString       date;
    size_t        size;
    TQSize         dims;
    ImageInfoList itemsList;
    TQDataStream   ds(data, IO_ReadOnly);

    while (!ds.atEnd())
    {
        ds >> imageID;
        ds >> albumID;
        ds >> name;
        ds >> date;
        ds >> size;
        ds >> dims;

        ImageInfo* info = new ImageInfo(imageID, albumID, name,
                                        TQDateTime::fromString(date, Qt::ISODate),
                                        size, dims);

        itemsList.append(info);
    }

    emit signalItemsInfo(itemsList);
}

}  // namespace Digikam
