/*
 * kcmreslisa.cpp
 *
 * Copyright (c) 2000-2002 Alexander Neundorf <neundorf@kde.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
#include "kcmreslisa.h"
#include "findnic.h"

#include <tqspinbox.h>
#include <tqcheckbox.h>
#include <tqvbox.h>
#include <tqpushbutton.h>
#include <tqgrid.h>
#include <tqlabel.h>
#include <tqtooltip.h>
#include <tqlayout.h>
#include <tqvbuttongroup.h>

#include <kdialogbase.h>
#include <tdelocale.h>
#include <tdemessagebox.h>
#include <krestrictedline.h>
#include <keditlistbox.h>

ResLisaSettings::ResLisaSettings(const TQString& config, TQWidget *parent)
: TDECModule(parent)
,m_config(config,false,true)
,m_tdeiolanConfig("tdeio_lanrc",false,true)
,m_advancedSettingsButton(0)
,m_suggestSettings(0)
,m_useNmblookup(0)
,m_pingNames(0)
,m_allowedAddresses(0)
,m_firstWait(0)
,m_secondScan(0)
,m_secondWait(0)
,m_updatePeriod(0)
,m_deliverUnnamedHosts(0)
,m_maxPingsAtOnce(0)
,m_reslisaAdvancedDlg(0)
{
   TQVBoxLayout *layout = new TQVBoxLayout(this, KDialog::marginHint(), KDialog::spacingHint());
   layout->setAutoAdd(true);
   
   TQVButtonGroup *gb=new TQVButtonGroup(i18n("Tell ResLISa Daemon How to Search for Hosts"),this);
   gb->setInsideSpacing(10);

   m_useNmblookup=new TQCheckBox(i18n("Send &NetBIOS broadcasts using &nmblookup"),gb);
   TQToolTip::add(m_useNmblookup,i18n("Only hosts running SMB servers will answer"));

   m_pingNames=new KEditListBox(i18n("A&dditionally Check These Hosts"),gb,"a",false, KEditListBox::Add|KEditListBox::Remove);
   TQToolTip::add(m_pingNames,i18n("The hosts listed here will be pinged."));

   TQHBox *hbox=new TQHBox(this);
   hbox->setSpacing(10);
   TQLabel *label=new TQLabel(i18n("&Trusted addresses:"),hbox);
   TQString comment = i18n("Usually your network address/subnet mask (e.g. 192.168.0.0/255.255.255.0;)");
   TQToolTip::add(label, comment);
   m_allowedAddresses=new KRestrictedLine(hbox,"a","0123456789./;");
   TQToolTip::add(m_allowedAddresses,comment);
   label->setBuddy(m_allowedAddresses);

    m_rlanSidebar = new TQCheckBox(i18n("Use &rlan:/ instead of lan:/ in Konqueror's navigation panel"), this);

   hbox = new TQHBox(this);
   m_suggestSettings=new TQPushButton(i18n("&Suggest Settings"),hbox);

   new TQWidget(hbox);

   m_advancedSettingsButton=new TQPushButton(i18n("Ad&vanced Settings"),hbox);


   m_reslisaAdvancedDlg=new KDialogBase(0,0,true,i18n("Advanced Settings for ResLISa"),KDialogBase::Close, KDialogBase::Close);
   connect(m_advancedSettingsButton,TQT_SIGNAL(clicked()),m_reslisaAdvancedDlg,TQT_SLOT(show()));

   TQVBox *vbox=m_reslisaAdvancedDlg->makeVBoxMainWidget();

   m_deliverUnnamedHosts=new TQCheckBox(i18n("Show &hosts without DNS names"),vbox);

   TQGrid *advGrid = new TQGrid(2, Qt::Horizontal, vbox);
   advGrid->setSpacing(10);

   label=new TQLabel(i18n("Host list update interval:"),advGrid);
   TQToolTip::add(label,i18n("Search hosts after this number of seconds"));
   m_updatePeriod=new TQSpinBox(30,1800,10,advGrid);
   m_updatePeriod->setSuffix(i18n(" sec"));
   TQToolTip::add(m_updatePeriod,i18n("Search hosts after this number of seconds"));

   m_secondScan=new TQCheckBox(i18n("Always check twice for hosts when searching"),advGrid);
   new TQWidget(advGrid);

   label=new TQLabel(i18n("Wait for replies from hosts after first scan:"),advGrid);
   TQToolTip::add(label,i18n("How long to wait for replies to the ICMP echo requests from hosts"));
   m_firstWait=new TQSpinBox(10,1000,50,advGrid);
   m_firstWait->setSuffix(i18n(" ms"));
   TQToolTip::add(m_firstWait,i18n("How long to wait for replies to the ICMP echo requests from hosts"));

   label=new TQLabel(i18n("Wait for replies from hosts after second scan:"),advGrid);
   TQToolTip::add(label,i18n("How long to wait for replies to the ICMP echo requests from hosts"));
   m_secondWait=new TQSpinBox(0,1000,50,advGrid);
   m_secondWait->setSuffix(i18n(" ms"));
   TQToolTip::add(m_secondWait,i18n("How long to wait for replies to the ICMP echo requests from hosts"));

   label=new TQLabel(i18n("Max. number of ping packets to send at once:"),advGrid);
   m_maxPingsAtOnce=new TQSpinBox(8,1024,5,advGrid);

   TQWidget *dummy=new TQWidget(advGrid);
   dummy->setMinimumHeight(10);

   connect(m_secondScan,TQT_SIGNAL(toggled(bool)),m_secondWait,TQT_SLOT(setEnabled(bool)));

   connect(m_allowedAddresses,TQT_SIGNAL(textChanged(const TQString&)),this,TQT_SIGNAL(changed()));

   connect(m_allowedAddresses,TQT_SIGNAL(returnPressed()),this,TQT_SIGNAL(changed()));

   connect(m_firstWait,TQT_SIGNAL(valueChanged(int)),this,TQT_SIGNAL(changed()));
   connect(m_secondWait,TQT_SIGNAL(valueChanged(int)),this,TQT_SIGNAL(changed()));
   connect(m_maxPingsAtOnce,TQT_SIGNAL(valueChanged(int)),this,TQT_SIGNAL(changed()));
   connect(m_secondScan,TQT_SIGNAL(toggled(bool)),this,TQT_SIGNAL(changed()));
   connect(m_deliverUnnamedHosts,TQT_SIGNAL(toggled(bool)),this,TQT_SIGNAL(changed()));
   connect(m_updatePeriod,TQT_SIGNAL(valueChanged(int)),this,TQT_SIGNAL(changed()));
   connect(m_pingNames,TQT_SIGNAL(changed()),this,TQT_SIGNAL(changed()));
   connect(m_useNmblookup,TQT_SIGNAL(toggled(bool)),this,TQT_SIGNAL(changed()));
   connect(m_suggestSettings,TQT_SIGNAL(clicked()),this,TQT_SLOT(suggestSettings()));
   connect(m_rlanSidebar,TQT_SIGNAL(clicked()),this,TQT_SIGNAL(changed()));

   load();
}

void ResLisaSettings::load()
{
   int secondWait=m_config.readNumEntry("SecondWait",-1);
   if (secondWait<0)
   {
      m_secondWait->setValue(300);
      m_secondScan->setChecked(FALSE);
      m_secondWait->setEnabled(FALSE);
   }
   else
   {
      m_secondWait->setValue(secondWait*10);
      m_secondScan->setChecked(TRUE);
      m_secondWait->setEnabled(TRUE);
   };
   m_deliverUnnamedHosts->setChecked(m_config.readNumEntry("DeliverUnnamedHosts",0));

   m_firstWait->setValue(m_config.readNumEntry("FirstWait",30)*10);
   m_maxPingsAtOnce->setValue(m_config.readNumEntry("MaxPingsAtOnce",256));
   m_updatePeriod->setValue(m_config.readNumEntry("UpdatePeriod",300));
   m_allowedAddresses->setText(m_config.readEntry("AllowedAddresses","192.168.0.0/255.255.255.0"));

   int i=m_config.readNumEntry("SearchUsingNmblookup",1);
   m_useNmblookup->setChecked(i!=0);
   m_pingNames->clear();
   m_pingNames->insertStringList(m_config.readListEntry("PingNames",';'));

   m_rlanSidebar->setChecked(m_tdeiolanConfig.readEntry("sidebarURL", "lan:/") == "rlan:/" ? true : false );
}

void ResLisaSettings::save()
{
   if (m_secondScan->isChecked())
      m_config.writeEntry("SecondWait",(m_secondWait->value()+5)/10);
   else
      m_config.writeEntry("SecondWait",-1);

   if (m_useNmblookup->isChecked())
      m_config.writeEntry("SearchUsingNmblookup",1);
   else
      m_config.writeEntry("SearchUsingNmblookup",0);

   if (m_deliverUnnamedHosts->isChecked())
      m_config.writeEntry("DeliverUnnamedHosts",1);
   else
      m_config.writeEntry("DeliverUnnamedHosts",0);

   m_config.writeEntry("FirstWait",(m_firstWait->value()+5)/10);
   m_config.writeEntry("MaxPingsAtOnce",m_maxPingsAtOnce->value());
   m_config.writeEntry("UpdatePeriod",m_updatePeriod->value());
   m_config.writeEntry("AllowedAddresses",m_allowedAddresses->text());
   //m_config.writeEntry("BroadcastNetwork",m_broadcastNetwork->text());
   TQStringList writeStuff;
   for (int i=0; i<m_pingNames->count(); i++)
      writeStuff.append(m_pingNames->text(i));
   m_config.writeEntry("PingNames",writeStuff,';');

   m_config.sync();

   m_tdeiolanConfig.writeEntry("sidebarURL", m_rlanSidebar->isChecked() ? "rlan:/" : "lan:/");
   m_tdeiolanConfig.sync();

}

void ResLisaSettings::suggestSettings()
{
   NICList* nics=findNICs();
   if (nics->count()==0)
   {
      //ok, easy one :-)
      KMessageBox::sorry(0,i18n("It appears you do not have any network interfaces installed on your system."));
      delete nics;
      return;
   }

   MyNIC *nic=nics->first();
   TQString address = nic->addr;
   TQString netmask = nic->netmask;
   m_allowedAddresses->setText(address+"/"+netmask+";");
   m_secondWait->setValue(0);
   m_secondScan->setChecked(FALSE);
   m_secondWait->setEnabled(FALSE);
   m_firstWait->setValue(300);
   m_maxPingsAtOnce->setValue(256);
   m_updatePeriod->setValue(300);
   m_useNmblookup->setChecked(true);
   if (nics->count()>1)
   {
      TQString msg(i18n("You have more than one network interface installed.<br>"
                       "Please make sure the suggested settings are correct.<br>"
                       "<br>The following interfaces were found:<br><br>"));
      //not that easy to handle
      for (MyNIC* tmp=nics->first(); tmp!=0; tmp=nics->next())
      {
         msg+="<b>"+tmp->name+": </b>"+tmp->addr+"/"+tmp->netmask+";<br>";
      }
      KMessageBox::information(0,TQString("<html>%1</html>").arg(msg));
   }
   KMessageBox::information(0,TQString("<html>%1</html>").arg(i18n("The ResLISa daemon is now configured "
                                                                  "correctly, hopefully.<br>Make sure that the reslisa binary is installed <i>suid root</i>.")));

   emit changed();
   delete nics;
}

#include "kcmreslisa.moc"

