/*
    This file is part of libkcal.

    Copyright (c) 2003 Cornelius Schumacher <schumacher@kde.org>
    Copyright (C) 2003-2004 Reinhold Kainhofer <reinhold@kainhofer.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/
/**
   @file calendarresources.h
   Provides a Calendar composed of several Calendar Resources.

   @author Cornelius Schumacher
   @author Reinhold Kainhofer
 */
#ifndef KCAL_CALENDARRESOURCES_H
#define KCAL_CALENDARRESOURCES_H

#include <tqintdict.h>
#include <tqmap.h>

#include "calendar.h"
#include "resourcecalendar.h"

#include "libkcal_export.h"

#include <tderesources/manager.h>

class TQWidget;

/**
   @namespace KCal
   Namespace KCal is for global classes, objects and/or functions in libkcal.
*/
namespace KCal {

class CalFormat;

/**
   @class CalendarResources

   This class provides a Calendar which is composed of other Calendars
   known as "Resources".

   Examples of Calendar Resources are:
     - Calendars stored as local ICS formatted files
     - a set of incidences (one-per-file) within a local directory
     - birthdays and anniversaries contained in an addressbook

*/
class LIBKCAL_EXPORT CalendarResources :
      public Calendar,
      public KRES::ManagerObserver<ResourceCalendar>
{
  Q_OBJECT
  
  public:
    /**
       @class DestinationPolicy
    */
    class DestinationPolicy
    {
      public:
        DestinationPolicy( CalendarResourceManager *manager,
                              TQWidget *parent = 0  ) :
          mManager( manager ), mParent( parent ) {}
        virtual ~DestinationPolicy() {}

        virtual ResourceCalendar *destination( Incidence *incidence ) = 0;
        virtual TQWidget *parent() { return mParent; }
        virtual void setParent( TQWidget *newparent ) { mParent = newparent; }
        bool hasCalendarResources();
      protected:
        CalendarResourceManager *resourceManager()
         { return mManager; }

      private:
        CalendarResourceManager *mManager;
        TQWidget *mParent;
    };

    /**
       @class StandardDestinationPolicy
    */
    class StandardDestinationPolicy : public DestinationPolicy
    {
      public:
        StandardDestinationPolicy( CalendarResourceManager *manager,
                              TQWidget *parent = 0  ) :
          DestinationPolicy( manager, parent ) {}
        virtual ~StandardDestinationPolicy() {}

        ResourceCalendar *destination( Incidence *incidence );

      private:
        class Private;
        Private *d;
    };

    /**
       @class AskDestinationPolicy
    */
    class AskDestinationPolicy : public DestinationPolicy
    {
      public:
        AskDestinationPolicy( CalendarResourceManager *manager,
                              TQWidget *parent = 0 ) :
          DestinationPolicy( manager, parent ) {}
        virtual ~AskDestinationPolicy() {}

        ResourceCalendar *destination( Incidence *incidence );

      private:
        class Private;
        Private *d;
    };

    /**
       @class Ticket
    */
    class Ticket
    {
        friend class CalendarResources;
      public:
        ResourceCalendar *resource() const
          { return mResource; }

      private:
        Ticket( ResourceCalendar *r ) : mResource( r ) {}

        ResourceCalendar *mResource;

        class Private;
        Private *d;
    };

    /**
       Construct CalendarResource object using a Time Zone and a Family name.

       @param timeZoneId is a string containing a Time Zone ID, which is
       assumed to be valid. The Time Zone Id is used to set the time zone
       for viewing Incidence dates.\n
       On some systems, /usr/share/zoneinfo/zone.tab may be available for
       reference.\n
       @e Example: "Europe/Berlin"

       @warning
       Do Not pass an empty timeZoneId string as this may cause unintended
       consequences when storing Incidences into the Calendar.

       @param family is any TQString representing a unique name.
    */
    CalendarResources(
      const TQString &timeZoneId,
      const TQString &family = TQString::fromLatin1( "calendar" ) );

    /**
       Destructor
    */
    ~CalendarResources();

    /**
       Loads all Incidences from the Resources.  The Resources must be added
       first using either readConfig(TDEConfig *config), which adds the system
       Resources, or manually using resourceAdded(ResourceCalendar *resource).
    */
    void load();

    /**
     * Reloads all incidences from all resources.
     * @param tz The timezone to set.
     * @return success or failure
     */
    bool reload( const TQString &tz );

    /**
       Clear out the current Calendar, freeing all used memory etc.
    */
    void close();

    /**
       Clear out the current Calendar, freeing all used memory etc.
    */
    void closeEvents();

    /**
       Clear out the current Calendar, freeing all used memory etc.
    */
    void closeTodos();

    /**
       Clear out the current Calendar, freeing all used memory etc.
    */
    void closeJournals();

    /**
       Save this Calendar.
       If the save is successfull, the Ticket is deleted.  Otherwise, the
       caller must release the Ticket with releaseSaveTicket() to abandon
       the save operation or call save() to try the save again.

       @param ticket is a pointer to the Ticket object.
       @param incidence is a pointer to the Incidence object.
       If incidence is null, save the entire Calendar (all Resources)
       else only the specified Incidence is saved.

       @return true if the save was successful; false otherwise.
    */
    virtual bool save( Ticket *ticket, Incidence *incidence = 0 );

    /**
       Sync changes in memory to persistant storage.
    */
    void save();

    /**
       Determine if the Calendar is currently being saved.

       @return true if the Calendar is currently being saved; false otherwise.
    */
    bool isSaving();

    /**
       Get the CalendarResourceManager used by this calendar.

       @return a pointer to the CalendarResourceManage.
    */
    CalendarResourceManager *resourceManager() const
      { return mManager; }

    /**
       Get the Resource associated with a specified Incidence.

       @param incidence is a pointer to an Incidence whose Resource
       is to be located.

       @return a pointer to the Resource containing the Incidence.
    */
    ResourceCalendar *resource( Incidence *incidence );

    /**
       Read the Resources settings from a config file.

       @param config The TDEConfig object which points to the config file.
       If no object is given (null pointer) the standard config file is used.

       @note Call this method <em>before</em> load().
    */
    void readConfig( TDEConfig *config = 0 );

    /**
       Set the destination policy such that Incidences are always added
       to the standard Resource.
    */
    void setStandardDestinationPolicy();

    /**
       Set the destination policy such that Incidences are added to a
       Resource which is queried.
    */
    void setAskDestinationPolicy();

    /**
       Returns the current parent for new dialogs. This is a bad hack, but we need
       to properly set the parent for the resource selection dialog. Otherwise
       the dialog will not be modal to the editor dialog in korganizer and
       the user can still work in the editor dialog (and thus crash korganizer).
       Afterwards we need to reset it (to avoid pointers to widgets that are
       already deleted) so we also need the accessor
    */
    TQWidget *dialogParentWidget();
    /**
       Set the widget parent for new dialogs. This is a bad hack, but we need
       to properly set the parent for the resource selection dialog. Otherwise
       the dialog will not be modal to the editor dialog in korganizer and
       the user can still work in the editor dialog (and thus crash korganizer).
    */
    void setDialogParentWidget( TQWidget *parent );

    /**
       Request ticket for saving the Calendar.  If a ticket is returned the
       Calendar is locked for write access until save() or releaseSaveTicket()
       is called.

       @param resource is a pointer to the ResourceCalendar object.

       @return a pointer to a Ticket object indicating that the Calendar
       is locked for write access; otherwise a null pointer.
    */
    Ticket *requestSaveTicket( ResourceCalendar *resource );

    /**
       Release the save Ticket. The Calendar is unlocked without saving.

       @param ticket is a pointer to a Ticket object.
    */
    virtual void releaseSaveTicket( Ticket *ticket );

    /**
       Add a Resource to the Calendar.
       This method must be public, because in-process added Resources
       do not emit the corresponding signal, so this method has to be
       called manually!

       @param resource is a pointer to the ResourceCalendar to add.
    */
    void resourceAdded( ResourceCalendar *resource );

// Incidence Specific Methods //

    /**
       Insert an Incidence into the Calendar.

       @param incidence is a pointer to the Incidence to insert.

       @return true if the Incidence was successfully inserted; false otherwise.
    */
    bool addIncidence( Incidence *incidence );

    /**
       Insert an Incidence into a Calendar Resource.

       @param incidence is a pointer to the Incidence to insert.
       @param resource is a pointer to the ResourceCalendar to be added to.

       @return true if the Incidence was successfully inserted; false otherwise.
       @deprecated use
       addIncidence(Incidence *,ResourceCalendar *,const TQString &) instead.
    */
    KDE_DEPRECATED bool addIncidence( Incidence *incidence, ResourceCalendar *resource );

    /**
       Insert an Incidence into a Calendar Resource.

       @param incidence is a pointer to the Incidence to insert.
       @param resource is a pointer to the ResourceCalendar to be added to.
       @param subresource is the subresource name, which may not be used
       by some calendar resources.

       @return true if the Incidence was successfully inserted; false otherwise.
    */
    bool addIncidence( Incidence *incidence,
                       ResourceCalendar *resource, const TQString &subresource );

    /**
       Flag that a change to a Calendar Incidence is starting.
       @param incidence is a pointer to the Incidence that will be changing.

       @return false if the resource could not be computed or if a ticket
       request fails; true otherwise.
    */
    KDE_DEPRECATED bool beginChange( Incidence *incidence );

    /**
       Flag that a change to a Calendar Incidence is starting.
       @param incidence is a pointer to the Incidence that will be changing.
       @param resource is a pointer to the ResourceCalendar that @p incidence
       belongs to; if this is 0 then the resource is queried via the
       DestinationPolicy.
       @param subresource is the @p Incidence subresource name, which may not
       be used by some calendar resources.

       @return false if the resource could not be computed or if a ticket
       request fails; true otherwise.
    */
    bool beginChange( Incidence *incidence, ResourceCalendar *resource, const TQString &subresource );

    /**
       Flag that a change to a Calendar Incidence has completed.
       @param incidence is a pointer to the Incidence that was changed.

       @return false if the resource could not be computed or if a ticket
       save fails; true otherwise.
    */
    KDE_DEPRECATED bool endChange( Incidence *incidence );

    /**
       Flag that a change to a Calendar Incidence has completed.
       @param incidence is a pointer to the Incidence that was changed.
       @param resource is a pointer to the ResourceCalendar that @p incidence
       belongs to; if this is 0 then the resource is queried via the
       DestinationPolicy.
       @param subresource is the @p Incidence subresource name, which may not
       be used by some calendar resources.

       @return false if the resource could not be computed or if a ticket
       save fails; true otherwise.
    */
    bool endChange( Incidence *incidence,
                    ResourceCalendar *resource, const TQString &subresource );

// Event Specific Methods //

    /**
       Insert an Event into the Calendar.

       @param event is a pointer to the Event to insert.

       @return true if the Event was successfully inserted; false otherwise.

       @note In most cases use
       addIncidence( Incidence *incidence ) instead.
    */
    bool addEvent( Event *event );

    /**
       Insert an Event into a Calendar Resource.

       @param event is a pointer to the Event to insert.
       @param resource is a pointer to the ResourceCalendar to be added to.

       @return true if the Event was successfully inserted; false otherwise.

       @deprecated use
       addIncidence(Incidence *,ResourceCalendar *,const TQString&) instead.
    */
    KDE_DEPRECATED bool addEvent( Event *event, ResourceCalendar *resource );

    /**
       Insert an Event into a Calendar Resource.

       @param event is a pointer to the Event to insert.
       @param resource is a pointer to the ResourceCalendar to be added to.
       @param subresource is the subresource name, which may not be used
       by some calendar resources.

       @return true if the Event was successfully inserted; false otherwise.

       @note In most cases use
       addIncidence(Incidence *,ResourceCalendar *,const TQString &) instead.
    */
    bool addEvent( Event *event, ResourceCalendar *resource, const TQString &subresource );

    /**
       Remove an Event from the Calendar.

       @param event is a pointer to the Event to remove.

       @return true if the Event was successfully removed; false otherwise.

       @note In most cases use
       deleteIncidence( Incidence *incidence) instead.
    */
    bool deleteEvent( Event *event );

    /**
       Return a sorted, unfiltered list of all Events.

       @param sortField specifies the EventSortField.
       @param sortDirection specifies the SortDirection.

       @return the list of all unfiltered Events sorted as specified.
    */
    Event::List rawEvents(
      EventSortField sortField = EventSortUnsorted,
      SortDirection sortDirection = SortDirectionAscending );

    /**
       Return an unfiltered list of all Events which occur on the given
       timestamp.

       @param qdt request unfiltered Event list for this TQDateTime only.

       @return the list of unfiltered Events occurring on the specified
       timestamp.
    */
    Event::List rawEventsForDate( const TQDateTime &qdt );

    /**
       Return an unfiltered list of all Events occurring within a date range.

       @param start is the starting date.
       @param end is the ending date.
       @param inclusive if true only Events which are completely included
       within the date range are returned.

       @return the list of unfiltered Events occurring within the specified
       date range.
    */
    Event::List rawEvents( const TQDate &start, const TQDate &end,
                           bool inclusive = false );

    /**
       Return a sorted, unfiltered list of all Events which occur on the given
       date.  The Events are sorted according to @a sortField and
       @a sortDirection.

       @param date request unfiltered Event list for this TQDate only.
       @param sortField specifies the EventSortField.
       @param sortDirection specifies the SortDirection.

       @return the list of sorted, unfiltered Events occurring on @a date.
    */
    Event::List rawEventsForDate(
      const TQDate &date,
      EventSortField sortField = EventSortUnsorted,
      SortDirection sortDirection = SortDirectionAscending );

    /**
       Returns the Event associated with the given unique identifier.

       @param uid is a unique identifier string.

       @return a pointer to the Event.
       A null pointer is returned if no such Event exists.
    */
    Event *event( const TQString &uid );

// Todo Specific Methods //

    /**
       Insert a Todo into a Calendar Resource.

       @param todo is a pointer to the Todo to insert.

       @return true if the Todo was successfully inserted; false otherwise.

       @note In most cases use
       addIncidence( Incidence *incidence ) instead.
    */
    bool addTodo( Todo *todo );

    /**
       Insert an Todo into a Calendar Resource.

       @param todo is a pointer to the Todo to insert.
       @param resource is a pointer to the ResourceCalendar to be added to.

       @return true if the Todo was successfully inserted; false otherwise.

       @deprecated use
       addIncidence(Incidence *,ResourceCalendar *,const TQString &) instead.
    */
    KDE_DEPRECATED bool addTodo( Todo *todo, ResourceCalendar *resource );

    /**
       Insert an Todo into a Calendar Resource.

       @param todo is a pointer to the Todo to insert.
       @param resource is a pointer to the ResourceCalendar to be added to.
       @param subresource is the subresource name, which may not be used
       by some calendar resources.

       @return true if the Todo was successfully inserted; false otherwise.

       @note In most cases use
       addIncidence(Incidence *, ResourceCalendar *,const TQString &) instead.
    */
    bool addTodo( Todo *todo, ResourceCalendar *resource, const TQString &subresource );

    /**
       Remove an Todo from the Calendar.

       @param todo is a pointer to the Todo to remove.

       @return true if the Todo was successfully removed; false otherwise.

       @note In most cases use
       deleteIncidence( Incidence *incidence ) instead.
    */
    bool deleteTodo( Todo *todo );

    /**
       Return a sorted, unfiltered list of all Todos for this Calendar.

       @param sortField specifies the TodoSortField.
       @param sortDirection specifies the SortDirection.

       @return the list of all unfiltered Todos sorted as specified.
    */
    Todo::List rawTodos( TodoSortField sortField = TodoSortUnsorted,
                         SortDirection sortDirection = SortDirectionAscending );

    /**
       Return an unfiltered list of all Todos which are due on the specified
       date.

       @param date request unfiltered Todos due on this TQDate.

       @return the list of unfiltered Todos due on the specified date.
    */
    Todo::List rawTodosForDate( const TQDate &date );

    /**
       Returns the Todo associated with the given unique identifier.

       @param uid is a unique identifier string.

       @return a pointer to the Todo.
       A null pointer is returned if no such Todo exists.
    */
    Todo *todo( const TQString &uid );

// Journal Specific Methods //

    /**
       Insert a Journal into the Calendar.

       @param journal is a pointer to the Journal to insert.

       @return true if the Journal was successfully inserted; false otherwise.

       @note In most cases use
       addIncidence( Incidence *incidence ) instead.
    */
    bool addJournal( Journal *journal );

    /**
       Insert a Journal into a Calendar Resource.

       @param journal is a pointer to the Journal to insert.
       @param resource is a pointer to the ResourceCalendar to be added to.

       @return true if the Journal was successfully inserted; false otherwise.

       @deprecated use
       addIncidence(Incidence *,ResourceCalendar *,const TQString &) instead.
    */
    KDE_DEPRECATED bool addJournal( Journal *journal, ResourceCalendar *resource );

    /**
       Insert a Journal into a Calendar Resource.

       @param journal is a pointer to the Journal to insert.
       @param resource is a pointer to the ResourceCalendar to be added to.
       @param subresource is the subresource name, which may not be used
       by some calendar resources.

       @return true if the Journal was successfully inserted; false otherwise.

       @note In most cases use
       addIncidence(Incidence *,ResourceCalendar *,const TQString &) instead.
    */
    bool addJournal( Journal *journal, ResourceCalendar *resource, const TQString &subresource );

    /**
       Remove a Journal from the Calendar.

       @param journal is a pointer to the Journal to remove.

       @return true if the Journal was successfully removed; false otherwise.

       @note In most cases use
       deleteIncidence( Incidence *incidence ) instead.
    */
    bool deleteJournal( Journal *journal );

    /**
       Return a sorted, unfiltered list of all Journals for this Calendar.

       @param sortField specifies the JournalSortField.
       @param sortDirection specifies the SortDirection.

       @return the list of all unfiltered Journals sorted as specified.
    */
    Journal::List rawJournals(
      JournalSortField sortField = JournalSortUnsorted,
      SortDirection sortDirection = SortDirectionAscending );

    /**
       Return an unfiltered list of all Journals for on the specifed date.

       @param date request unfiltered Journals for this TQDate only.

       @return the list of unfiltered Journals for the specified date.
    */
    Journal::List rawJournalsForDate( const TQDate &date );

    /**
       Returns the Journal associated with the given unique identifier.

       @param uid is a unique identifier string.

       @return a pointer to the Journal.
       A null pointer is returned if no such Journal exists.
    */
    Journal *journal( const TQString &uid );

// Alarm Specific Methods //

    /**
       Return a list of Alarms within a time range for this Calendar.

       @param from is the starting timestamp.
       @param to is the ending timestamp.

       @return the list of Alarms for the for the specified time range.
    */
    Alarm::List alarms( const TQDateTime &from, const TQDateTime &to );

    /**
       Return a list of Alarms that occur before the specified timestamp.

       @param to is the ending timestamp.

       @return the list of Alarms occurring before the specified TQDateTime.
    */
    Alarm::List alarmsTo( const TQDateTime &to );

    /**
     * Set the viewing time zone, which requires that all resources are saved,
     * and then reloaded such that the event times are re-interpreted in the new
     * timezone. Note that the absolute times of events do not change with this.
     * If you want to change the times of the contents of the resources, use
     * setTimeZoneId
     */
    void setTimeZoneIdViewOnly( const TQString& tz );

  //issue 2508
    bool hasCalendarResources();
  signals:
    /**
       Signal that the Resource has been modified.
    */
    void signalResourceModified( ResourceCalendar *resource );

    /**
       Signal that an Incidence has been inserted to the Resource.
    */
    void signalResourceAdded( ResourceCalendar *resource );

    /**
       Signal that an Incidence has been removed from the Resource.
    */
    void signalResourceDeleted( ResourceCalendar *resource );

    /**
       Signal an error message.
    */
    void signalErrorMessage( const TQString &err );

  protected:
    void connectResource( ResourceCalendar *resource );
    void resourceModified( ResourceCalendar *resource );
    void resourceDeleted( ResourceCalendar *resource );

    /**
       Let CalendarResource subclasses set the Time Zone ID.

       First parameter is a string containing a Time Zone ID, which is
       assumed to be valid. On some systems, /usr/share/zoneinfo/zone.tab
       may be available for reference.\n
       @e Example: "Europe/Berlin"

       @warning
       Do Not pass an empty timeZoneId string as this may cause unintended
       consequences when storing Incidences into the Calendar.
    */
    virtual void doSetTimeZoneId( const TQString &timeZoneId );

    /**
       Increment the number of times this Resource has been changed by 1.

       @param resource is a pointer to the ResourceCalendar to be counted.

       @return the new number of times this Resource has been changed.
    */
    int incrementChangeCount( ResourceCalendar *resource );

    /**
       Decrement the number of times this Resource has been changed by 1.

       @param resource is a pointer to the ResourceCalendar to be counted.

       @return the new number of times this Resource has been changed.
    */
    int decrementChangeCount( ResourceCalendar *resource );

  protected slots:
    void slotLoadError( ResourceCalendar *resource, const TQString &err );
    void slotSaveError( ResourceCalendar *resource, const TQString &err );

    /**
      All addIncidence( Incidence * ), addTodo( Todo * ) addEvent( Event * )
      and addJournal( Journal * ) calls made between beginAddingIncidences()
      and endAddingIncidences() will only ask the user to choose a resource once.
      @since 4.4
    */
    void beginAddingIncidences();

    /**
      @see beginAddingIncidences()
      @since 4.4
    */
    void endAddingIncidences();

  private:

    /**
       Initialize the Resource object with starting values.
    */
    void init( const TQString &family );

    bool mOpen;

    KRES::Manager<ResourceCalendar>* mManager;
    TQMap <Incidence*, ResourceCalendar*> mResourceMap;

    DestinationPolicy *mDestinationPolicy;
    StandardDestinationPolicy *mStandardPolicy;
    AskDestinationPolicy *mAskPolicy;
    bool mPendingDeleteFromResourceMap;

    TQMap<ResourceCalendar *, Ticket *> mTickets;
    TQMap<ResourceCalendar *, int> mChangeCounts;

    class Private;
    Private *d;
};

}

#endif
