/*
    This file is part of libtdepim.

    Copyright (c) 2003 Cornelius Schumacher <schumacher@kde.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#include "tdeconfigwizard.h"

#include <tdelocale.h>
#include <kdebug.h>
#include <tdeconfigskeleton.h>
#include <tdemessagebox.h>
#include <tdeapplication.h>

#include <tqlistview.h>
#include <tqlayout.h>
#include <tqtimer.h>

TDEConfigWizard::TDEConfigWizard( TQWidget *parent,
                              char *name, bool modal )
  : KDialogBase( TreeList, i18n("Configuration Wizard"), Ok|Cancel, Ok, parent,
                 name, modal ),
    mPropagator( 0 ), mChangesPage( 0 )
{
  init();
}

TDEConfigWizard::TDEConfigWizard( TDEConfigPropagator *propagator, TQWidget *parent,
                              char *name, bool modal )
  : KDialogBase( TreeList, i18n("Configuration Wizard"), Ok|Cancel, Ok, parent,
                 name, modal ),
    mPropagator( propagator ), mChangesPage( 0 )
{
  init();
}

TDEConfigWizard::~TDEConfigWizard()
{
  delete mPropagator;
}

void TDEConfigWizard::init()
{
  connect( this, TQT_SIGNAL( aboutToShowPage( TQWidget * ) ),
           TQT_SLOT( slotAboutToShowPage( TQWidget * ) ) );

  TQTimer::singleShot( 0, this, TQT_SLOT( readConfig() ) );
}

void TDEConfigWizard::setPropagator( TDEConfigPropagator *p )
{
  mPropagator = p;
}

void TDEConfigWizard::slotAboutToShowPage( TQWidget *page )
{
  if ( page == mChangesPage ) {
    updateChanges();
  }
}

TQFrame *TDEConfigWizard::createWizardPage( const TQString &title )
{
  return addPage( title );
}

void TDEConfigWizard::setupRulesPage()
{
  TQFrame *topFrame = addPage( i18n("Rules") );
  TQVBoxLayout *topLayout = new TQVBoxLayout( topFrame );

  mRuleView = new TQListView( topFrame );
  topLayout->addWidget( mRuleView );

  mRuleView->addColumn( i18n("Source") );
  mRuleView->addColumn( i18n("Target") );
  mRuleView->addColumn( i18n("Condition") );

  updateRules();
}

void TDEConfigWizard::updateRules()
{
  if ( !mPropagator ) {
    kdError() << "TDEConfigWizard: No TDEConfigPropagator set." << endl;
    return;
  }

  mRuleView->clear();

  TDEConfigPropagator::Rule::List rules = mPropagator->rules();
  TDEConfigPropagator::Rule::List::ConstIterator it;
  for( it = rules.begin(); it != rules.end(); ++it ) {
    TDEConfigPropagator::Rule r = *it;
    TQString source = r.sourceFile + "/" + r.sourceGroup + "/" +
                     r.sourceEntry;
    TQString target = r.targetFile + "/" + r.targetGroup + "/" +
                     r.targetEntry;
    TQString condition;
    TDEConfigPropagator::Condition c = r.condition;
    if ( c.isValid ) {
      condition = c.file + "/" + c.group + "/" + c.key + " = " + c.value;
    }
    new TQListViewItem( mRuleView, source, target, condition );
  }
}

void TDEConfigWizard::setupChangesPage()
{
  TQFrame *topFrame = addPage( i18n("Changes") );
  TQVBoxLayout *topLayout = new TQVBoxLayout( topFrame );

  mChangeView = new TQListView( topFrame );
  topLayout->addWidget( mChangeView );

  mChangeView->addColumn( i18n("Action") );
  mChangeView->addColumn( i18n("Option") );
  mChangeView->addColumn( i18n("Value") );
  mChangeView->setSorting( -1 );

  mChangesPage = topFrame;
}

void TDEConfigWizard::updateChanges()
{
  kdDebug() << "TDEConfigWizard::updateChanges()" << endl;

  if ( !mPropagator ) {
    kdError() << "TDEConfigWizard: No TDEConfigPropagator set." << endl;
    return;
  }

  usrWriteConfig();

  mPropagator->updateChanges();

  mChangeView->clear();

  TDEConfigPropagator::Change::List changes = mPropagator->changes();
  TDEConfigPropagator::Change *c;
  for( c = changes.first(); c; c = changes.next() ) {
    new TQListViewItem( mChangeView, mChangeView->lastItem(), c->title(), c->arg1(), c->arg2() );
  }
}

void TDEConfigWizard::readConfig()
{
  kdDebug() << "TDEConfigWizard::readConfig()" << endl;

  int result = KMessageBox::warningContinueCancel( this,
      i18n("Please make sure that the programs which are "
           "configured by the wizard do not run in parallel to the wizard; "
           "otherwise, changes done by the wizard could be lost."),
      i18n("Warning"), i18n("Run Wizard Now"), "warning_running_instances" );
  if ( result != KMessageBox::Continue ) kapp->quit();

  usrReadConfig();
}

void TDEConfigWizard::slotOk()
{
  TQString error = validate();
  if ( error.isNull() ) {
    usrWriteConfig();

    if ( !mPropagator ) {
      kdError() << "TDEConfigWizard: No TDEConfigPropagator set." << endl;
      return;
    } else {
      if ( mPropagator->skeleton() ) {
        mPropagator->skeleton()->writeConfig();
      }
      mPropagator->commit();
    }

    accept();
  } else {
    KMessageBox::sorry( this, error );
  }
}

#include "tdeconfigwizard.moc"
