/*
 * Copyright (C) 2004, Mart Kelder (mart.kde@hccnet.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "tdeio_subjects.h"

#include "tdeio.h"
#include "tdeio_single_subject.h"
#include "tdeio_proto.h"
#include "mailsubject.h"

#include <tdeio/global.h>
#include <tdeio/scheduler.h>
#include <kdebug.h>

#include <tqptrlist.h>
#include <tqvaluelist.h>
#include <tqstring.h>

TDEIO_Subjects::TDEIO_Subjects( TQObject * parent, const char * name )
	: TQObject( parent, name ),
	_protocol( 0 ),
	_slave( 0 ),
	_valid( true )
{
	_jobs = new TQPtrList<TDEIO_Single_Subject>;
	_kurl = new KURL;
	_metadata = new TDEIO::MetaData;
	
	_jobs->setAutoDelete( true );
}

TDEIO_Subjects::~TDEIO_Subjects( )
{
	delete _jobs;
	delete _kurl;
	delete _metadata;
	_protocol = 0;
}

void TDEIO_Subjects::doReadSubjects( KKioDrop *drop )
{
	TQValueList<KKioDrop::FileInfo>::ConstIterator it;
	TQValueList<KKioDrop::FileInfo>::ConstIterator end_it = drop->_mailurls->end();
	
	_tdeio = drop;
	_protocol = _tdeio->_protocol;
	*_kurl = *_tdeio->_kurl;
	*_metadata = *_tdeio->_metadata;

	if( _jobs->count() > 0 )
		kdWarning() << i18n( "Already a slave pending." ) << endl;
		
	_jobs->clear( );
	
	//Open connection
	getConnection( );
	
	//Open jobs for easy item in the list
	for( it = _tdeio->_mailurls->begin(); it != end_it; it++ )
		startJob( (*it).name, (*it).size );
	
	//close connection for trivial situations (empty list)
	disConnect( true );
	
	//passing number of subjects for progress bar.
	_tdeio->emitReadSubjectsTotalSteps( _jobs->count() );
}

void TDEIO_Subjects::getConnection( )
{
	KURL kurl = *_kurl;
	TDEIO::MetaData metadata = *_metadata;

	if( _slave )
	{
		TDEIO::Scheduler::disconnectSlave( _slave );
		_slave = 0;
	}
	
	if( _protocol->connectionBased( ) )
	{
		_protocol->readSubjectConnectKURL( kurl, metadata );
		
		if( kurl.port() == 0 )
			kurl.setPort( _protocol->defaultPort( _tdeio->_ssl ) );
		
		if( ! ( _slave = TDEIO::Scheduler::getConnectedSlave( kurl, metadata ) ) )
		{
			kdWarning() << i18n( "Not able to open a tdeio-slave for %1." ).arg( _protocol->configName() );
			_tdeio->emitShowPassivePopup( i18n( "Not able to open a tdeio-slave for %1." ).arg( _protocol->configName() ) );
			_valid = false;
			_tdeio->emitValidChanged();
			_slave = 0;
			_tdeio->emitReadSubjectsReady( false );
			return;
		}
	}
}

void TDEIO_Subjects::startJob( const TQString &name, const long size )
{
	KURL kurl = *_kurl;
	TDEIO::MetaData metadata = *_metadata;
	TDEIO_Single_Subject *subject;
	
	kurl = name;
	
	_protocol->readSubjectKURL( kurl, metadata );
	
	if( kurl.port() == 0 )
		kurl.setPort( _protocol->defaultPort( _tdeio->_ssl ) );
	
	subject = new TDEIO_Single_Subject( this, name.latin1(), kurl, metadata, _protocol, _slave, name, size );
	
	connect( subject, TQT_SIGNAL( readSubject( KornMailSubject* ) ), this, TQT_SLOT( slotReadSubject( KornMailSubject* ) ) );
	connect( subject, TQT_SIGNAL( finished( TDEIO_Single_Subject* ) ), this, TQT_SLOT( slotFinished( TDEIO_Single_Subject* ) ) );
	
	_jobs->append( subject );
}

void TDEIO_Subjects::disConnect( bool result )
{
	if( _jobs->isEmpty() )
	{
		if( _slave )
		{
			TDEIO::Scheduler::disconnectSlave( _slave );
			_slave = 0;
		}
		_tdeio->emitReadSubjectsReady( result );
	}
}

void TDEIO_Subjects::cancelled( )
{
	_jobs->clear();
	//_slave died in cancelJob with is by called from the destructor of TDEIO_Single_Subject,
	//withs is by called by _jobs->clear because autoRemove equals true.
	_slave = 0;
	disConnect( false );
}

void TDEIO_Subjects::slotReadSubject( KornMailSubject* subject )
{
	_valid = true;
	_tdeio->emitValidChanged();
	subject->setMailDrop( _tdeio );
	_tdeio->emitReadSubjectsRead( subject );
}

void TDEIO_Subjects::slotFinished( TDEIO_Single_Subject* item )
{
	//Remove sender.... I didn't know of the computer gonna like me, but it seems he does :)
	_jobs->remove( item );
	
	_tdeio->emitReadSubjectsProgress( _jobs->count( ) );
	
	disConnect( true ); //Only works when all jobs are finished.
}

#include "tdeio_subjects.moc"
