#! /usr/bin/env python
#
# python.prov.py - find rpm provides for python modules
#
# Copyright (C) 2004,2005  Andrey Orlov <cray@altlinux.ru>
# Copyright (C) 2006       Ivan Fedorov <ns@altlinux.ru>
# Copyright (C) 2007       Alex V. Myltsev <avm@altlinux.ru>
# Copyright (C) 2008,2009  Alexey Tourbin <at@altlinux.ru>
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

from __future__ import generators
import sys, os
from os.path import basename, dirname

verbosity = os.getenv('RPM_SCRIPTS_DEBUG', 0)
prog = 'python.prov'

def normpath(f) :
    if not f.startswith('/') :
        f = os.getcwd() + f
    return os.path.normpath(f)

def get_buildroot() :
    buildroot = os.getenv('RPM_BUILD_ROOT','')
    if buildroot in ['','/'] :
        return None
    return normpath(buildroot)

buildroot = get_buildroot()

def get_path() :
    "get the list of standard python directories"
    # we assume that the first sys.path element is "" which stands for current directory,
    # and that sys.path entries are already normalized.
    path = sys.path[1:]
    if buildroot :
        # it is possible that sys.path already has RPM_BUILD_ROOT entries
        # e.g. if we build python itself and RPM_PYTHON is redefined
        l = len(buildroot)
        def strip_buildroot(p) :
            if p.startswith(buildroot+'/') :
                return p[l:]
            return p
        path = [ strip_buildroot(p) for p in path ]
    # handle RPM_PYTHON_LIB_PATH which can be exported with rpm macros
    rpm_path = os.getenv('RPM_PYTHON_LIB_PATH','').split()
    rpm_path = filter(lambda p: p.startswith('/'), rpm_path)
    path = rpm_path + path
    # now we can prepend RPM_BUILD_ROOT
    if buildroot :
        br_path = [ buildroot + p for p in path ]
        path = br_path + path
    return path

def get_pth_path() :
    "get the list of additional python directories from site-packages/*.pth"
    # cf. site.py:addsitedir()
    def glob_pth(dir) :
        try :
            files = os.listdir(dir)
        except :
            return []
        return [ f for f in files if f.endswith('.pth') ]
    def read_pth(f) :
        try :
            fp = open(f)
        except :
            return []
        subdirs = []
        for line in fp :
            if line.startswith("#"):
                continue
            if line.startswith("import"):
                continue
            line = line.rstrip()
            if len(line) == 0:
                continue
            subdirs.append(line)
        return subdirs
    # okay, here we go
    sitedirs = [ p for p in get_path() if p.endswith('/site-packages') ]
    pth_path = []
    for sitedir in sitedirs :
        for pth in glob_pth(sitedir) :
            for subdir in read_pth(sitedir+'/'+pth) :
                dir = normpath(sitedir+'/'+subdir)
                pth_path.append(dir)
    if buildroot :
        l = len(buildroot)
        def strip_buildroot(p) :
            if p.startswith(buildroot+'/') :
                return p[l:]
            return p
        pth_path = [ strip_buildroot(p) for p in pth_path ]
        br_pth_path = [ buildroot + p for p in pth_path ]
        pth_path = br_pth_path + pth_path
    #print >>sys.stderr, "%s: pth_path=" % prog, pth_path
    return pth_path

path = get_path() + get_pth_path()
path = [ p for p in path if os.path.exists(p) ]
# it is essential to sort path entries descending by their length,
# so that "most specific" entries are tried first; e.g.
# /usr/lib/python2.5/site-packages should go before /usr/lib/python2.5
path.sort( lambda a,b: len(b)-len(a) )
# debug
if __name__ == '__main__' and verbosity > 1 :
    print >>sys.stderr, "%s: path=" % prog, path

def explode(f) :
    "split filename into std-dir + subdir/basename"
    for p in path :
        if f.startswith(p+'/') :
            l = len(p)
            return f[:l],f[l+1:]
    return None, None

def prov1db(d,b) :
    "list of modules provided by the file, split into std-dir + subdir/basename"
    # Python syntax does not allow modules with dashes.
    # Also note that this will exclude e.g. b="site-packages/foo.py".
    if b.find("-") >= 0 :
        return
    m,e = os.path.splitext(b)
    if not e in ['.py','.so'] :
        return
    if not '/' in m :
        # the file is a simple module
        if m == '__init__' :
            return
        prov = [ m ]
        if e == '.so' and len(m) > 6 and m[-6:] == 'module' :
            # e.g. rpmmodule.so provides rpm
            prov.append(m[0:-6])
        return prov
    # hierarchical module
    parts = m.split('/')
    if parts[-1] == '__init__' :
        # e.g. foo/bar/__init__.py provides foo.bar
        return [ '.'.join(parts[:-1]) ]
    # for/bar.py provides foo.bar if foo/__init__.py is present
    if os.path.exists(d+'/'+dirname(b)+'/__init__.py') :
        m = '.'.join(parts)
        prov = [ m ]
        if e == '.so' and len(m) > 6 and m[-6:] == 'module' :
            prov.append(m[0:-6])
        return prov

def prov1(f) :
    "returns list of modules provided by the file"
    d,b = explode(f)
    if not d :
        return
    prov = prov1db(d,b)
    # Sometimes a module can be loaded by its full name, e.g. PIL.Image,
    # while PIL.pth adds PIL subdir to std-dirs, so that the module can be
    # loaded simply as Image.  We provide limited support for this feature.
    # Note that, since PIL.pth provides the longest std-dir, the b=Image case
    # has just been handled.  Now we check d to see if we need to recombine
    # std-dir + subdir/basename and run prov1db(d,b) again.
    if os.path.exists(d+'/__init__.py') and dirname(d) in path :
        b = basename(d)+'/'+b
        d = dirname(d)
        xprov = prov1db(d,b)
        if xprov :
            if prov :
                prov += xprov
            else :
                prov = xprov
    return prov

def verbose(msg) :
    if verbosity > 0 :
        print >>sys.stderr, "%s: %s" % (prog,msg)

def provides(f,m) :
    prefix = "python%u.%u" % sys.version_info[0:2]
    prov = "%s(%s)" % (prefix,m)
    verbose("%s provides %s" % (f,prov))
    print prov

def main(files, provides=provides, verbose=verbose) :
    for f in files :
        prov = prov1(f)
        if prov :
            for p in prov :
                provides(f,p)
        else :
            verbose("%s provides nothing" % f)

if __name__ == '__main__':
    files = sys.argv[1:] or [ x.strip() for x in sys.stdin.readlines() ]
    main(files)

# ex: set ts=8 sts=4 sw=4 et:
