
//OpenSCADA system file: ttransports.h
/***************************************************************************
 *   Copyright (C) 2003-2010 by Roman Savochenko                           *
 *   rom_as@oscada.org, rom_as@fromru.com                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; version 2 of the License.               *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef TTRANSPORTS_H
#define TTRANSPORTS_H

#define STR_VER		5		//TransportS type modules version
#define STR_ID		"Transport"

#include <string>

#include "tbds.h"

using std::string;

namespace OSCADA
{

//************************************************
//* TTransportIn				 *
//************************************************
class TTipTransport;

class TTransportIn : public TCntrNode, public TConfig
{
    public:
	//Methods
	TTransportIn( const string &id, const string &db, TElem *el );
	virtual ~TTransportIn( );

	TCntrNode &operator=( TCntrNode &node );

	const string &id( )	{ return mId; }
	string workId( );
	string name( );
	string dscr( )		{ return mDscr; }
	string addr( )		{ return mAddr; }
	string protocolFull( )	{ return mProt; }
	string protocol( );
	virtual string getStatus( );

	bool toStart( ) 	{ return mStart; }
	bool startStat( )	{ return run_st; }

	string DB( )		{ return mDB; }
	string tbl( );
	string fullDB( )	{ return DB()+'.'+tbl(); }

	void setName( const string &inm )		{ mName = inm; modif(); }
	void setDscr( const string &idscr )		{ mDscr = idscr; modif(); }
	virtual void setAddr( const string &addr )	{ mAddr = addr; modif(); }
	void setProtocolFull( const string &prt )	{ mProt = prt; modif(); }
	void setToStart( bool val )			{ mStart = val; modif(); }

	void setDB( const string &vl )			{ mDB = vl; modifG(); }

	virtual void start( )	{ };
	virtual void stop( )	{ };

	TTipTransport &owner( );

    protected:
	//Methods
	void cntrCmdProc( XMLNode *opt );	//Control interface command process

	void preEnable( int flag );
	void postDisable( int flag );		//Delete all DB if flag 1

	void load_( );
	void save_( );

	//Attributes
	bool	run_st;

    private:
	//Methods
	const string &nodeName( )	{ return mId; }

	//Attributes
	string	&mId;
	string	&mName;
	string	&mDscr;
	string	&mAddr;
	string	&mProt;
	bool	&mStart;
	string	mDB;
};

//************************************************
//* TTransportOut				 *
//************************************************
class TTransportOut : public TCntrNode, public TConfig
{
    public:
	//Methods
	TTransportOut( const string &id, const string &db, TElem *el );
	virtual ~TTransportOut( );

	TCntrNode &operator=( TCntrNode &node );

	const string &id( )	{ return mId; }
	string	workId( );
	string	name( );
	string	dscr( )		{ return mDscr; }
	string	addr( )		{ return mAddr; }
	int	prm1( )		{ return mPrm1; }
	int	prm2( )		{ return mPrm2; }
	bool	toStart( )	{ return mStart; }
	bool	startStat( )	{ return run_st; }
	virtual	string getStatus( );

	string DB( )		{ return mDB; }
	string tbl( );
	string fullDB( )	{ return DB()+'.'+tbl(); }

	void setName( const string &inm )		{ mName = inm; modif(); }
	void setDscr( const string &idscr )		{ mDscr = idscr; modif(); }
	virtual void setAddr( const string &addr )	{ mAddr = addr; modif(); }
	void setPrm1( int vl )				{ mPrm1 = vl; }
	void setPrm2( int vl )				{ mPrm2 = vl; }
	void setToStart( bool val )			{ mStart = val; modif(); }

	void setDB( const string &vl )			{ mDB = vl; modifG(); }

	virtual void start( )			{ };
	virtual void stop( )			{ };

	virtual int messIO( const char *obuf, int len_ob, char *ibuf = NULL, int len_ib = 0, int time = 0, bool noRes = false )
	{ return 0; }

	void messProtIO( XMLNode &io, const string &prot );

	TTipTransport &owner( );

	Res &nodeRes( )					{ return nRes; }

    protected:
	//Methods
	void cntrCmdProc( XMLNode *opt );	//Control interface command process

	void preEnable( int flag );
	void postDisable( int flag );		//Delete all DB if flag 1

	TVariant objFuncCall( const string &id, vector<TVariant> &prms, const string &user );

	void load_( );
	void save_( );

	//Attributes
	bool	run_st;

    private:
	//Methods
	const string &nodeName( )	{ return mId; }

	//Attributes
	string	&mId;
	string	&mName;
	string	&mDscr;
	string	&mAddr;
	bool	&mStart;
	string	mDB;

	//> Reserve parameters
	int	mPrm1, mPrm2;
	Res	nRes;
};

//************************************************
//* TTipTransport				 *
//************************************************
class TTransportS;

class TTipTransport: public TModule
{
    public:
	//Methods
	TTipTransport( const string &id );
	virtual ~TTipTransport( );

	//> Input transports
	void inList( vector<string> &list )			{ chldList(mIn,list); }
	bool inPresent( const string &name )			{ return chldPresent(mIn,name); }
	void inAdd( const string &name, const string &db = "*.*" );
	void inDel( const string &name, bool complete = false )	{ chldDel(mIn,name,-1,complete); }
	AutoHD<TTransportIn> inAt( const string &name )		{ return chldAt(mIn,name); }

	//> Output transports
	void outList( vector<string> &list )			{ chldList(mOut,list); }
	bool outPresent( const string &name )			{ return chldPresent(mOut,name); }
	void outAdd( const string &name, const string &idb = "*.*" );
	void outDel( const string &name, bool complete = false ){ chldDel(mOut,name,-1,complete); }
	AutoHD<TTransportOut> outAt( const string &name )	{ return chldAt(mOut,name); }

	TTransportS &owner( );

    protected:
	//Methods
	void cntrCmdProc( XMLNode *opt );	//Control interface command process

	virtual TTransportIn  *In( const string &name, const string &db )
	{ throw TError(nodePath().c_str(),_("Input transport no support!")); }
	virtual TTransportOut *Out( const string &name, const string &db )
	{ throw TError(nodePath().c_str(),_("Output transport no support!")); }

    private:
	//Attributes
	int	mIn, mOut;
};

//************************************************
//* TTransportS					 *
//************************************************
class TTransportS : public TSubSYS
{
    public:
	//Data
	class ExtHost
	{
	    public:
		//Methods
		ExtHost( const string &iuser_open, const string &iid, const string &iname,
			    const string &itransp, const string &iaddr, const string &iuser, const string &ipass ) :
		    user_open(iuser_open), id(iid), name(iname), transp(itransp), addr(iaddr),
		    user(iuser), pass(ipass), link_ok(false) { }

		//Attributes
		string	user_open;	//User which open remote host
		string	id;		//External host id
		string	name;		//Name
		string	transp;		//Connect transport
		string	addr;		//External host address
		string	user;		//External host user
		string	pass;		//External host password
		bool	link_ok;	//Link OK
	};

	//Methods
	TTransportS( );
	~TTransportS( );

	int subVer( )			{ return STR_VER; }
	void inTrList( vector<string> &ls );
	void outTrList( vector<string> &ls );

	//> External hosts
	bool sysHost( )			{ return sys_host; }
	void setSysHost( bool vl )	{ sys_host = vl; }
	string extHostsDB( );
	void extHostList( const string &user, vector<string> &list );
	bool extHostPresent( const string &user, const string &id );
	AutoHD<TTransportOut> extHost( TTransportS::ExtHost host, const string &pref = "" );
	ExtHost extHostGet( const string &user, const string &id );
	void extHostSet( const ExtHost &host );
	void extHostDel( const string &user, const string &id );

	//> Request to remote or local OpenSCADA control interface
	int cntrIfCmd( XMLNode &node, const string &senderPref, const string &user = "" );

	void subStart( );
	void subStop( );

	TElem &inEl( )			{ return el_in; }
	TElem &outEl( ) 		{ return el_out; }

	AutoHD<TTipTransport> at( const string &iid )	{ return modAt(iid); }

    protected:
	void load_( );
	void save_( );

    private:
	//Methods
	string optDescr( );
	void cntrCmdProc( XMLNode *opt );       //Control interface command process

	//Attributes
	bool	sys_host;
	TElem	el_in, el_out, el_ext;

	Res	extHostRes;             //External hosts resource
	vector<ExtHost> extHostLs;      //External hosts list
};

}

#endif // TTRANSPORTS_H
