
//OpenSCADA system file: xml.cpp
/***************************************************************************
 *   Copyright (C) 2003-2010 by Roman Savochenko                           *
 *   rom_as@oscada.org, rom_as@fromru.com                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; version 2 of the License.               *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "terror.h"
#include "tmess.h"
#include "xml.h"

using namespace OSCADA;

//*************************************************
//* XMLNode                                       *
//*************************************************
XMLNode::XMLNode(const XMLNode &nd) : mName("nd"), mText(""), mParent(NULL)
{
    this->operator=(nd);
}

XMLNode &XMLNode::operator=(const XMLNode &prm)
{
    //> Delete self children and attributes
    mAttr.clear();
    mPrcInstr.clear();
    for(unsigned i_ch = 0; i_ch < mChildren.size(); i_ch++)
	delete mChildren[i_ch];
    mChildren.clear();

    //> Copy params (name,text, attributes and instructions)
    setName( prm.name() );
    setText( prm.text() );
    vector<string> ls;
    prm.attrList(ls);
    for(unsigned i_a = 0; i_a < ls.size(); i_a++)
	setAttr(ls[i_a],prm.attr(ls[i_a]));
    prm.prcInstrList(ls);
    for(unsigned i_p = 0; i_p < ls.size(); i_p++)
	setPrcInstr(ls[i_p],prm.prcInstr(ls[i_p]));

    //> Recursive copy children
    for(unsigned i_ch = 0; i_ch < prm.childSize(); i_ch++)
	*childAdd() = *prm.childGet(i_ch);

    return *this;
}

void XMLNode::childAdd( XMLNode *n )
{
    if(!n) return;
    mChildren.push_back(n);
    n->mParent = this;
}

XMLNode* XMLNode::childAdd( const string &name )
{
    XMLNode *n = new XMLNode(name);
    childAdd(n);

    return n;
}

void XMLNode::childDel( const unsigned id )
{
    if(id >= childSize()) throw TError("XMLNode",_("Child %d is not present."),id);
    delete mChildren[id];
    mChildren.erase(mChildren.begin()+id);
}

void XMLNode::childDel( XMLNode *nd )
{
    for(unsigned i_ch = 0; i_ch < mChildren.size(); i_ch++)
	if(mChildren[i_ch] == nd)
	{
	    delete mChildren[i_ch];
	    mChildren.erase( mChildren.begin()+i_ch );
	    break;
	}
}

void XMLNode::childClear( const string &name )
{
    for(unsigned i_ch = 0; i_ch < mChildren.size(); )
	if(name.empty() || mChildren[i_ch]->name() == name) childDel(i_ch);
	else i_ch++;
}

int XMLNode::childIns( unsigned id, XMLNode * n )
{
    if(!n) return -1;

    if(id > childSize()) id = childSize();
    mChildren.insert(mChildren.begin()+id, n);
    n->mParent = this;

    return id;
}

XMLNode* XMLNode::childIns( unsigned id, const string &name )
{
    XMLNode *n = new XMLNode( name );
    childIns(id,n);

    return n;
}

XMLNode* XMLNode::childGet( const int index, bool noex ) const
{
    if(index >= 0 && index < (int)childSize()) return mChildren[index];
    if(noex) return NULL;
    throw TError("XMLNode",_("Child %d is not present."),index);
}

XMLNode* XMLNode::childGet( const string &name, const int numb, bool noex ) const
{
    for(int i_ch = 0, i_n = 0; i_ch < (int)childSize(); i_ch++)
	if(childGet(i_ch)->name() == name && i_n++ == numb)
	    return childGet(i_ch);

    if(noex) return NULL;
    throw TError("XMLNode",_("Child %s:%d is not found!"),name.c_str(),numb);
}

XMLNode* XMLNode::childGet( const string &attr, const string &val, bool noex ) const
{
    for(unsigned i_f = 0; i_f < childSize(); i_f++)
	if(childGet(i_f)->attr(attr) == val) return childGet(i_f);

    if(noex) return NULL;
    throw TError("XMLNode",_("Child with attribut %s=%s is not present."),attr.c_str(),val.c_str());
}

void XMLNode::attrList( vector<string> & list ) const
{
    list.clear();
    for(unsigned i_a = 0; i_a < mAttr.size(); i_a++)
	list.push_back(mAttr[i_a].first);
}

XMLNode* XMLNode::attrDel( const string &name )
{
    for(unsigned i_a = 0; i_a < mAttr.size(); i_a++)
	if(mAttr[i_a].first == name)
	{ mAttr.erase(mAttr.begin()+i_a); break; }

    return this;
}

void XMLNode::attrClear( )
{
    mAttr.clear();
}

string XMLNode::attr( const string &name ) const
{
    for(unsigned i_a = 0; i_a < mAttr.size(); i_a++)
	if(mAttr[i_a].first == name) return mAttr[i_a].second;

    return "";
}

XMLNode* XMLNode::setAttr( const string &name, const string &val )
{
    for(unsigned i_a = 0; i_a < mAttr.size(); i_a++)
	if(mAttr[i_a].first == name)
	{
	    mAttr[i_a].second = val;
	    return this;
	}

    mAttr.push_back(pair<string,string>(name,val));

    return this;
}

void XMLNode::prcInstrList( vector<string> & list ) const
{
    list.clear();
    for(unsigned i_p = 0; i_p < mPrcInstr.size(); i_p++)
	list.push_back(mPrcInstr[i_p].first);
}

void XMLNode::prcInstrDel( const string &target )
{
    for(unsigned i_p = 0; i_p < mPrcInstr.size(); i_p++)
	if(mPrcInstr[i_p].first == target)
	{ mPrcInstr.erase(mPrcInstr.begin()+i_p); return; }
}

void XMLNode::prcInstrClear( )
{
    mPrcInstr.clear();
}

string XMLNode::prcInstr( const string &target ) const
{
    for(unsigned i_p = 0; i_p < mPrcInstr.size(); i_p++)
	if(mPrcInstr[i_p].first == target) return mPrcInstr[i_p].second;

    return "";
}

XMLNode* XMLNode::setPrcInstr( const string &target, const string &val )
{
    for(unsigned i_p = 0; i_p < mPrcInstr.size(); i_p++)
	if(mPrcInstr[i_p].first == target)
	{
	    mPrcInstr[i_p].second = val;
	    return this;
	}

    mPrcInstr.push_back(pair<string,string>(target,val));

    return this;
}

XMLNode* XMLNode::clear()
{
    attrClear();
    setText("");
    prcInstrClear();
    childClear();

    return this;
}

string XMLNode::save( unsigned flg )
{
    string xml;
    xml.reserve(10000);

    if(flg&XMLHeader)
    {
	xml += "<?xml version='1.0' encoding='UTF-8' ?>";
	if(flg&XMLNode::BrClosePast) xml += "\n";
    }

    saveNode(flg,xml);
    return xml;
}



void XMLNode::saveNode( unsigned flg, string &xml )
{
    xml.append((flg&XMLNode::BrOpenPrev) ? "\n<" : "<");
    if(flg&XMLNode::MissTagEnc) xml.append(name());
    else encode(name(), xml);

    for(unsigned i_a = 0; i_a < mAttr.size(); i_a++)
    {
	xml.append(" ");
	if(flg&XMLNode::MissAttrEnc) xml.append(mAttr[i_a].first);
	else encode(mAttr[i_a].first, xml);
	xml.append("=\"");
	encode(Mess->codeConvOut("UTF-8",mAttr[i_a].second), xml);
	xml.append("\"");
    }

    if(childEmpty() && text().empty() && mPrcInstr.empty()) xml.append((flg&(XMLNode::BrOpenPast|XMLNode::BrClosePast)) ? "/>\n" : "/>");
    else
    {
	xml.append((flg&XMLNode::BrOpenPast) ? ">\n" : ">");
	//> Save text
	if(!text().empty())
	{
	    encode(Mess->codeConvOut("UTF-8",text()), xml, true);
	    xml.append(flg&XMLNode::BrTextPast ? "\n" : "");
	}
	//> Save process instructions
	for(unsigned i_p = 0; i_p < mPrcInstr.size(); i_p++)
	    xml.append("<?"+mPrcInstr[i_p].first+" "+Mess->codeConvOut("UTF-8",mPrcInstr[i_p].second)+(flg&XMLNode::BrPrcInstrPast?"?>\n":"?>"));
	//> Save included childs
	for(unsigned i_c = 0; i_c < childSize(); i_c++) childGet(i_c)->saveNode(flg,xml);
	//> Close tag
	xml.append("</");
	if(flg&XMLNode::MissTagEnc) xml.append(name() );
	else encode(name(), xml);
	xml.append(flg&XMLNode::BrClosePast ? ">\n" : ">");
    }
}

void XMLNode::encode( const string &s, string &rez, bool text ) const
{
    const char *replStr = NULL;
    for(unsigned i_sz = 0, f_pos = 0; true; )
    {
	switch(s[i_sz])
	{
	    case '>': replStr = "&gt;"; break;
	    case '<': replStr = "&lt;"; break;
	    case '"': replStr = "&quot;"; break;
	    case '&': replStr = "&amp;"; break;
	    case '\'': replStr = "&#039;"; break;
	    case '\n': if( !text ) replStr = "&#010;"; break;
	}
	i_sz++;
	if(replStr)
	{
	    if((i_sz-1) > f_pos) rez.append(s,f_pos,i_sz-f_pos-1);
	    rez.append(replStr);
	    replStr = NULL;
	    f_pos = i_sz;
	}
	if(i_sz >= s.size())
	{
	    if(i_sz > f_pos) rez.append(s,f_pos,i_sz-f_pos);
	    break;
	}
    }
}

void XMLNode::load( const string &s )
{
    clear();

#if HAVE_EXPAT_H
    XML_Parser p = XML_ParserCreate("UTF-8");
    if(!p) throw TError("XMLNode",_("Couldn't allocate memory for parser."));

    XML_SetElementHandler(p, start_element, end_element);
    XML_SetCharacterDataHandler(p, characters);
    XML_SetProcessingInstructionHandler(p, instrHandler);
    XML_SetUserData(p, this);

    XMLNode *lstParent = mParent;
    mParent = NULL;

    if(!XML_Parse(p, s.data(), s.size(), true))
    {
	int cL = XML_GetCurrentLineNumber(p);
	string xmlErr = XML_ErrorString(XML_GetErrorCode(p));
	XML_ParserFree(p);
	mParent = lstParent;
	throw TError("XMLNode",_("Parse error at line %d --- %s. Source string: '%s'"), cL, xmlErr.c_str(), ((s.size()>1024)?s.substr(0,1024)+"...":s).c_str());
    }
    XML_ParserFree(p);
    mParent = lstParent;

#else
    LoadCtx ctx(s);
    loadNode(ctx);

#endif
}

#if HAVE_EXPAT_H
//> Parse/load XML attributes
void XMLNode::start_element( void *data, const char *el, const char **attr )
{
    const char *a_n, *a_v;
    XMLNode *p = (XMLNode*)data;
    XMLNode *n = p;

    if(p->mParent) n = p->mParent->childAdd();

    n->setName(el);
    while(*attr)
    {
	a_n = *attr++; a_v = *attr++;
	n->mAttr.push_back(pair<string,string>(a_n,Mess->codeConvIn("UTF-8",a_v)));
    }

    p->node_stack.push_back(n);
    p->mParent = n;
}

void XMLNode::end_element( void *data, const char *el )
{
    XMLNode *p = (XMLNode*)data;

    //> Remove spaces from end of text
    int i_ch;
    for(i_ch = p->mParent->mText.size()-1; i_ch >= 0; i_ch--)
	if(!isspace(p->mParent->mText[i_ch]))
	    break;
    //> Encode text
    p->mParent->setText(Mess->codeConvIn("UTF-8",p->mParent->mText.substr(0,i_ch+1)));

    if(!p->node_stack.empty()) p->node_stack.pop_back();
    if(p->node_stack.empty()) p->mParent = NULL;
    else p->mParent = p->node_stack[p->node_stack.size()-1];
}

void XMLNode::characters( void *userData, const XML_Char *s, int len )
{
    XMLNode *p = ((XMLNode*)userData)->mParent;
    //if( !len )	return;
    if(p->mText.size()) p->mText.append(s,len);
    else
	for(int i_ch = 0; i_ch < len; i_ch++)
	{
	    if(isspace(s[i_ch])) continue;
	    p->mText.assign(s+i_ch,len-i_ch);
	    break;
	}
}

void XMLNode::instrHandler( void *userData, const XML_Char *target, const XML_Char *data )
{
    ((XMLNode*)userData)->mParent->mPrcInstr.push_back(pair<string,string>(target,Mess->codeConvIn("UTF-8",data)));
}

#else
unsigned XMLNode::loadNode( LoadCtx &ctx, unsigned pos )
{
    bool initTag = true;
    unsigned cpos, bpos, tpos, bufp = 0;

nextTag:
    //> Find for a tag start symbol
    for( ; pos < ctx.vl.size() && ctx.vl[pos] != '<'; pos++)
    {
	if(initTag) continue;
	if(bufp || !isspace(ctx.vl[pos]) || mText.size())
	{
	    ctx.buf[bufp++] = (ctx.vl[pos] != '&') ? ctx.vl[pos] : parseEntity(ctx,pos);
	    if(bufp == ctx.bufSz) { mText.append(ctx.buf,bufp); bufp = 0; }
	}
    }
    if((pos+2) >= ctx.vl.size()) throw TError("XMLNode",_("Unexpected end. Pos: %d"),pos);
    cpos = pos;

    switch(ctx.vl[pos+1])
    {
	case '!':	// Comment - special part
	    //> Comment block
	    if(ctx.vl.compare(pos,4,"<!--") == 0)
	    {
		for(cpos += 4; ctx.vl.compare(cpos,3,"-->") != 0; cpos++)
		    if(cpos >= ctx.vl.size()) throw TError("XMLNode",_("No comment block end. Pos: %d"),pos);
		pos = cpos+3;
	    }
	    //> Special "DOCTYPE" block
	    else if(ctx.vl.compare(pos,10,"<!DOCTYPE ") == 0)
	    {
		//>> Find subblock
		for(cpos += 10; ctx.vl[cpos] != '['; cpos++)
		    if(cpos >= ctx.vl.size()) throw TError("XMLNode",_("Unfinished '!DOCTYPE' block. Pos: %d"),pos);
		    else if(ctx.vl[cpos] == '>') { pos = cpos+1; goto nextTag; }
		//>> Process entities container
		for(cpos += 1; ctx.vl[cpos] != ']'; cpos++)
		    if(cpos >= ctx.vl.size()) throw TError("XMLNode",_("Unfinished '!DOCTYPE [ ]' container. Pos: %d"),pos);
		    else if(ctx.vl.compare(cpos,9,"<!ENTITY ") == 0)
		    {
			for(cpos += 9; parseAttr(ctx,cpos,0); )
			    if(ctx.aVl.size()) ctx.ent.insert(pair<string,char>(ctx.aNm,ctx.aVl[0]));
			while(isspace(ctx.vl[cpos])) cpos++;
			if(ctx.vl[cpos] != '>')	throw TError("XMLNode",_("Unexpected or error end tag. Pos: %d"),cpos);
			cpos++;
		    }
		cpos++;
		while(isspace(ctx.vl[cpos])) cpos++;
		if(ctx.vl[cpos] != '>')	throw TError("XMLNode",_("Unexpected or error end tag. Pos: %d"),cpos);
		pos = cpos+1;
	    }
	    else throw TError("XMLNode",_("Unfinished start comment or unknown block. Pos: %d"),pos);
	    goto nextTag;
	case '?':	// Program block
	    for(bpos = cpos+2, tpos = bpos; ctx.vl.compare(tpos,2,"?>") != 0; tpos++)
		if(tpos >= ctx.vl.size()) throw TError("XMLNode",_("Unexpected end. Pos: %d"),pos);
	    //> Get program block's name
	    //>>> Get tag name
	    cpos = bpos;
	    while(cpos < tpos && !isspace(ctx.vl[cpos])) cpos++;
	    if(cpos < tpos)
	    {
		string nm(ctx.vl,bpos,cpos-bpos);
		//>> Pass spaces
		while(isspace(ctx.vl[cpos])) cpos++;
		//> Place program block
		if(!initTag) mPrcInstr.push_back(pair<string,string>(nm,ctx.vl.substr(cpos,tpos-cpos)));
		//> Process specific block <?xml ?>
		if(nm == "xml")
		    while(parseAttr(ctx,cpos))
			if(ctx.aNm.compare("encoding") == 0) ctx.enc.assign(ctx.aVl);
	    }
	    pos = tpos+2;
	    goto nextTag;
	case '/':	//> End tag
	    if(ctx.vl.compare(cpos+2,mName.size(),mName) == 0)
	    {
		cpos += 2+mName.size();
		while(isspace(ctx.vl[cpos])) cpos++;
		if(ctx.vl[cpos] == '>')
		{
		    if(bufp) mText.append(ctx.buf,bufp);
		    if(mText.size())
		    {
			//> Remove spaces from end of text
			int i_ch = mText.size()-1;
			while(i_ch >= 0 && isspace(mText[i_ch])) i_ch--;
			mText = Mess->codeConvIn(ctx.enc,mText.substr(0,i_ch+1));
		    }
		    return cpos+1;
		}
	    }
	    throw TError("XMLNode",_("Unexpected or error end tag. Pos: %d"),cpos);
    }
    //> Process for standard XML node
    //>> It is me node
    if(initTag)
    {
	bpos = cpos+1;
	//>>> Get tag name
	for(cpos = bpos; !isspace(ctx.vl[cpos]) && ctx.vl[cpos] != '>' && ctx.vl[cpos] != '/'; cpos++)
	    if(cpos >= ctx.vl.size()) throw TError("XMLNode",_("Unexpected end. Pos: %d"),pos);
	mName.assign(ctx.vl,bpos,cpos-bpos);
	initTag = false;
	//>>> Process tag attributes
	while(parseAttr(ctx,cpos))
	    mAttr.push_back(pair<string,string>(ctx.aNm,ctx.aVl.size()?Mess->codeConvIn(ctx.enc,ctx.aVl):string("")));
	//>>> Pass spaces
	while(isspace(ctx.vl[cpos])) cpos++;
	//>>> Process close tag or the tag content
	if(ctx.vl[cpos] == '>') { pos = cpos+1; goto nextTag; }
	else if(ctx.vl.compare(cpos,2,"/>") == 0) return cpos+2;
	throw TError("XMLNode",_("Start tag error. Pos: %d"),cpos);
    }
    //>> New XML node create
    else
    {
	if(bufp) { mText.append(ctx.buf,bufp); bufp = 0; }
	pos = childAdd()->loadNode(ctx,pos-1);
	goto nextTag;
    }

    return cpos;
}

bool XMLNode::parseAttr( LoadCtx &ctx, unsigned &pos, char sep )
{
    //> Get attribute name
    //>> Pass spaces
    while(isspace(ctx.vl[pos])) pos++;
    if(!isalpha(ctx.vl[pos])) return false;

    unsigned bpos = pos;
    for( ; !isspace(ctx.vl[pos]) && ctx.vl[pos] != '='; pos++)
	if(pos >= ctx.vl.size()) throw TError("XMLNode",_("Unexpected end. Pos: %d"),pos);
    ctx.aNm.assign(ctx.vl,bpos,pos-bpos);
    //> Get symbol '='
    //>> Pass spaces
    while(isspace(ctx.vl[pos])) pos++;
    if(sep)
    {
	if(ctx.vl[pos] != sep) throw TError("XMLNode",_("Unfinished attribute. Pos: %d"),bpos);
	pos++;
    }
    //> Get symbol "'" or '"'
    //>> Pass spaces
    while(isspace(ctx.vl[pos])) pos++;
    if(ctx.vl[pos] != '\'' && ctx.vl[pos] != '"') throw TError("XMLNode",_("Unfinished attribute. Pos: %d"),bpos);
    char brc = ctx.vl[pos];
    //> Get value
    unsigned cb = 0;
    ctx.aVl.clear();
    for(pos++; ctx.vl[pos] != brc; pos++)
    {
	if(pos >= ctx.vl.size()) throw TError("XMLNode",_("Unexpected end. Pos: %d"),pos);
	ctx.buf[cb++] = (ctx.vl[pos] != '&') ? ctx.vl[pos] : parseEntity(ctx,pos);
	if(cb == ctx.bufSz) { ctx.aVl.append(ctx.buf,cb); cb = 0; }
    }
    if(cb) ctx.aVl.append(ctx.buf,cb);
    pos++;

    return true;
}

char XMLNode::parseEntity( LoadCtx &ctx, unsigned &rpos )
{
    //> Check for standard entities
    if(ctx.vl.compare(rpos,5,"&amp;") == 0)	{ rpos += 4; return '&'; }
    if(ctx.vl.compare(rpos,4,"&lt;") == 0)	{ rpos += 3; return '<'; }
    if(ctx.vl.compare(rpos,4,"&gt;") == 0)	{ rpos += 3; return '>'; }
    if(ctx.vl.compare(rpos,6,"&apos;") == 0)	{ rpos += 5; return '\''; }
    if(ctx.vl.compare(rpos,6,"&quot;") == 0)	{ rpos += 5; return '"'; }
    //> Check for code entities
    if((rpos+3) < ctx.vl.size() && ctx.vl[rpos+1] == '#')
    {
	if(ctx.vl[rpos+2] == 'X' || ctx.vl[rpos+2] == 'x')
	{
	    rpos += 3;
	    unsigned nBeg = rpos;
	    while(isxdigit(ctx.vl[rpos])) rpos++;
	    if(ctx.vl[rpos] != ';') throw TError("XMLNode",_("Entity error. Pos: %d"),nBeg-3);
	    return (char)strtol(ctx.vl.data()+nBeg,NULL,16);
	}
	else
	{
	    rpos += 2;
	    unsigned nBeg = rpos;
	    while(isdigit(ctx.vl[rpos])) rpos++;
	    if(ctx.vl[rpos] != ';') throw TError("XMLNode",_("Entity error. Pos: %d"),nBeg-2);
	    return (char)atoi(ctx.vl.data()+nBeg);
	}
    }
    //> Check for loaded entities
    if(ctx.ent.size())
    {
	rpos += 1;
	unsigned nBeg = rpos;
	for( ; ctx.vl[rpos] != ';'; rpos++)
	    if(rpos >= ctx.vl.size()) throw TError("XMLNode",_("Entity error. Pos: %d"),nBeg-1);
	map<string,char>::iterator ient = ctx.ent.find(ctx.vl.substr(nBeg,rpos-nBeg));
	if(ient != ctx.ent.end()) return ient->second;
    }

    throw TError("XMLNode",_("Unknown entity. Pos: %d"),rpos);
}


//*************************************************
//* XMLNode::LoadCtx                              *
//*************************************************
XMLNode::LoadCtx::LoadCtx( const string &ivl ) : enc("UTF-8"), bufSz(1000)
{
    buf = (char*)malloc(bufSz);
    vl = ivl+char(0);
}

XMLNode::LoadCtx::~LoadCtx( )
{
    free(buf);
}
#endif
