/*
 * hotplug.c
 *
 * A version of /sbin/hotplug that is not a script.
 *
 * Copyright (C) 2001 Greg Kroah-Hartman <greg@kroah.com>
 *
 *	This program is free software; you can redistribute it and/or modify it
 *	under the terms of the GNU General Public License as published by the
 *	Free Software Foundation version 2 of the License.
 *
 *	This program is distributed in the hope that it will be useful, but
 *	WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *	General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License along
 *	with this program; if not, write to the Free Software Foundation, Inc.,
 *	675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Hacked up to read the various modules.*map files and for use
 * in BusyBox by Erik Andersen <andersen@codepoet.org>
 *
 */


#include <errno.h>
#include <stdio.h>
#include <stdarg.h>
#include <stddef.h>
#include <stdlib.h>
#include <string.h>
#include <syslog.h>
#include <unistd.h>
#include <sys/param.h>
#include <sys/utsname.h>

#define LINELENGTH 8192


/* Enable if not used as a busybox applet */
#if 1
static const char * const app_name = "hotplug";
static void verror_msg(const char *s, va_list p)
{
	fflush(stdout);
	fprintf(stderr, "\n\n\n%s: ", app_name);
	vfprintf(stderr, s, p);
}
extern void bb_error_msg(const char *s, ...)
{
	va_list p;

	va_start(p, s);
	verror_msg(s, p);
	va_end(p);
	fprintf(stderr, "\n\n\n");
}

extern void bb_error_msg_and_die(const char *s, ...)
{
	va_list p;

	va_start(p, s);
	verror_msg(s, p);
	va_end(p);
	fprintf(stderr, "\n\n\n");
	exit(EXIT_FAILURE);
}

static void vperror_msg(const char *s, va_list p)
{
	int err = errno;

	if (s == 0)
		s = "";
	verror_msg(s, p);
	if (*s)
		s = ": ";
	fprintf(stderr, "%s%s\n", s, strerror(err));
	fprintf(stderr, "\n\n\n");
}

extern void bb_perror_msg(const char *s, ...)
{
	va_list p;

	va_start(p, s);
	vperror_msg(s, p);
	va_end(p);
}

extern void bb_perror_msg_and_die(const char *s, ...)
{
	va_list p;

	va_start(p, s);
	vperror_msg(s, p);
	va_end(p);
	exit(EXIT_FAILURE);
}
static const char * const memory_exhausted = "memory exhausted";
extern void *xmalloc(size_t size)
{
	void *ptr = malloc(size);

	if (!ptr)
		bb_error_msg_and_die((char*)memory_exhausted);
	return ptr;
}

extern void *xrealloc(void *old, size_t size)
{
	void *ptr;

	/* SuS2 says "If size is 0 and ptr is not a null pointer, the
	 * object pointed to is freed."  Do that here, in case realloc
	 * returns a NULL, since we don't want to choke in that case. */
	if (size==0 && old) {
		free(old);
		return NULL;
	}

	ptr = realloc(old, size);
	if (!ptr)
		bb_error_msg_and_die((char*)memory_exhausted);
	return ptr;
}
#define hotplug_main	main
#endif

struct subsystem {
    const char * name;
    int (* handler) (void);
};

extern int log_message (int level, const char *fmt, ...)  __attribute__ ((__format__(__printf__, 2, 3)));
extern int split_3values (const char *string, int base, unsigned int * value1,
	unsigned int * value2, unsigned int * value3);
extern int split_2values (const char *string, int base, unsigned int * value1,
	unsigned int * value2);
extern int call_subsystem (const char *string, struct subsystem *subsystem);
extern int load_module (const char *module_name);
extern int usb_handler (void);
extern int pci_handler (void);
extern int ieee1394_handler (void);

#define ADD_STRING	"add"
#define REMOVE_STRING	"remove"


//#define DEBUG 1
#ifdef DEBUG
#include <syslog.h>
	#define dbg(format, arg...) do { log_message (LOG_DEBUG, __FUNCTION__ ": " format, ## arg); } while (0)
#else
	#define dbg(format, arg...) do { } while (0)
#endif


int cpu_handler (void)
{
    /* Not handled */
    return 0;
}

int net_handler (void)
{
    /* Not handled */
    return 0;
}

int dock_handler (void)
{
    /* Not handled */
    return 0;
}

static struct subsystem main_subsystem[] = {
    { "pci",	    pci_handler },
    { "usb",	    usb_handler },
    { "ieee1394",   ieee1394_handler },
    { "cpu",	    cpu_handler },
    { "net",	    net_handler },
    { "dock",	    dock_handler },
    { NULL,	    NULL }
};

int hotplug_main(int argc, char *argv[])
{
    if (argc != 2) {
	dbg ("unknown number of arguments");
	return 1;
    }
    /* pass control to the subsystem specified by argv[1]. */
    return call_subsystem (argv[1], main_subsystem);
}



/* The ieee1394 functions */
#define IEEE1394_MATCH_VENDOR_ID	0x0001
#define IEEE1394_MATCH_MODEL_ID		0x0002
#define IEEE1394_MATCH_SPECIFIER_ID	0x0004
#define IEEE1394_MATCH_VERSION		0x0008

struct ieee1394_module_map {
	const char * module_name;
	unsigned int match_flags;
	unsigned int vendor_id;
	unsigned int model_id;
	unsigned int specifier_id;
	unsigned int version;
} __attribute__ ((packed));
static struct ieee1394_module_map *ieee1394_module_map;

void ieee1394_read_pcimap(void)
{
    FILE *ieee1394map_file;
    struct utsname utsname;
    struct ieee1394_module_map *entry = ieee1394_module_map;
    char *prevmodule = "";
    char filename[MAXPATHLEN];
    char line[LINELENGTH];
    char module[LINELENGTH];
    int count, mapsize;

    if (uname(&utsname) < 0) {
	perror("uname");
	exit(1);
    }
    sprintf(filename, "/lib/modules/%s/modules.ieee1394map", utsname.release);
    if ((ieee1394map_file = fopen(filename, "r")) == NULL) {
	bb_perror_msg_and_die("Could not open %s", filename);
    }

    count = mapsize = 0;
    while(fgets(line, LINELENGTH, ieee1394map_file) != NULL) {

	if (line[0] == '#')
	    continue;

	count++;
	if (count >= mapsize) {
	    mapsize+=100;
	    ieee1394_module_map = xrealloc(ieee1394_module_map, (sizeof(struct ieee1394_module_map) * mapsize));
	    entry = ieee1394_module_map + count - 1;
	    memset(entry, 0, (sizeof(struct ieee1394_module_map) * 100));
	}

	if (sscanf(line, "%s 0x%x 0x%x 0x%x 0x%x 0x%x",
		    module,
		    &entry->match_flags, &entry->vendor_id,
		    &entry->model_id, &entry->specifier_id,
		    &entry->version) != 6) {
	    bb_error_msg("modules.ieee1394map unparsable line: %s.\n", line);
	    free(entry);
	    continue;
	}

	/* Optimize memory allocation a bit, in case someday we
	   have Linux systems with ~100,000 modules.  It also
	   allows us to just compare pointers to avoid trying
	   to load a module twice. */
	if (strcmp(module, prevmodule) != 0) {
	    prevmodule = xmalloc(strlen(module)+1);
	    strcpy(prevmodule, module);
	}
	entry->module_name = prevmodule;
    }
    fclose(ieee1394map_file);
}
static int ieee1394_match(unsigned int vendor_id, unsigned int specifier_id, unsigned int version)
{
    int i;
    int retval;

    dbg ("vendor_id = %x, specifier_id = %x, version = %x",
	    vendor_id, specifier_id, version);

    for (i = 0; ieee1394_module_map[i].module_name != NULL; ++i) {
	dbg ("looking at %s, match_flags = %x",
		ieee1394_module_map[i].module_name,
		ieee1394_module_map[i].match_flags);
	if ((ieee1394_module_map[i].match_flags & IEEE1394_MATCH_VENDOR_ID) &&
		(ieee1394_module_map[i].vendor_id != vendor_id)) {
	    dbg ("vendor check failed %x != %x",
		    ieee1394_module_map[i].vendor_id,
		    vendor_id);
	    continue;
	}
	if ((ieee1394_module_map[i].match_flags & IEEE1394_MATCH_SPECIFIER_ID) &&
		(ieee1394_module_map[i].specifier_id != specifier_id)) {
	    dbg ("specifier_id check failed %x != %x",
		    ieee1394_module_map[i].specifier_id,
		    specifier_id);
	    continue;
	}
	if ((ieee1394_module_map[i].match_flags & IEEE1394_MATCH_VERSION) &&
		(ieee1394_module_map[i].version != version)) {
	    dbg ("version check failed %x != %x",
		    ieee1394_module_map[i].version,
		    version);
	    continue;
	}
	/* found one! */
	dbg ("loading %s", ieee1394_module_map[i].module_name);
	retval = load_module (ieee1394_module_map[i].module_name);
	if (retval)
	    return retval;
    }

    return 0;
}


static int ieee1394_add (void)
{
    char *vendor_env;
    char *specifier_env;
    char *version_env;
    int error;
    unsigned int vendor_id;
    unsigned int specifier_id;
    unsigned int version;

    ieee1394_read_pcimap();

    vendor_env = getenv ("VENDOR_ID");
    specifier_env = getenv ("SPECIFIER_ID");
    version_env = getenv ("VERSION");
    if ((vendor_env == NULL) ||
	    (specifier_env == NULL) ||
	    (version_env == NULL)) {
	dbg ("missing an environment variable, aborting.");
	return 1;
    }
    vendor_id = strtoul (vendor_env, NULL, 16);
    specifier_id = strtoul (specifier_env, NULL, 16);
    version = strtoul (version_env, NULL, 16);

    error = ieee1394_match (vendor_id, specifier_id, version);

    return error;
}


static int ieee1394_remove (void)
{
    /* right now we don't do anything here :) */
    return 0;
}


static struct subsystem ieee1394_subsystem[] = {
    { ADD_STRING, ieee1394_add },
    { REMOVE_STRING, ieee1394_remove },
    { NULL, NULL }
};


int ieee1394_handler (void)
{
    char * action;
    action = getenv ("ACTION");
    dbg ("action = %s", action);
    if (action == NULL) {
	dbg ("missing ACTION environment variable, aborting.");
	return 1;
    }
    return call_subsystem (action, ieee1394_subsystem);
}

/* The pci functions */
#define PCI_ANY				0xffffffff

struct pci_module_map {
    const char * module_name;
    unsigned int vendor;
    unsigned int device;
    unsigned int subvendor;
    unsigned int subdevice;
    unsigned int class;
    unsigned int class_mask;
} __attribute__ ((packed));
static struct pci_module_map *pci_module_map;

void pci_read_pcimap(void)
{
    FILE *pcimap_file;
    struct utsname utsname;
    struct pci_module_map *entry = pci_module_map;
    unsigned int driver_data;
    char *prevmodule = "";
    char filename[MAXPATHLEN];
    char line[LINELENGTH];
    char module[LINELENGTH];
    int count, mapsize;

    if (uname(&utsname) < 0) {
	perror("uname");
	exit(1);
    }
    sprintf(filename, "/lib/modules/%s/modules.pcimap", utsname.release);
    if ((pcimap_file = fopen(filename, "r")) == NULL) {
	bb_perror_msg_and_die("Could not open %s", filename);
    }

    count = mapsize = 0;
    while(fgets(line, LINELENGTH, pcimap_file) != NULL) {

	if (line[0] == '#')
	    continue;

	count++;
	if (count >= mapsize) {
	    mapsize+=100;
	    pci_module_map = xrealloc(pci_module_map, (sizeof(struct pci_module_map) * mapsize));
	    entry = pci_module_map + count - 1;
	    memset(entry, 0, (sizeof(struct pci_module_map) * 100));
	}
	entry = pci_module_map + count - 1;

	if (sscanf(line, "%s 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x",
		    module,
		    &entry->vendor, &entry->device,
		    &entry->subvendor, &entry->subdevice,
		    &entry->class, &entry->class_mask,
		    &driver_data) != 8) {
	    bb_error_msg("modules.pcimap unparsable line: %s.\n", line);
	    free(entry);
	    continue;
	}

	/* Optimize memory allocation a bit, in case someday we
	   have Linux systems with ~100,000 modules.  It also
	   allows us to just compare pointers to avoid trying
	   to load a module twice. */
	if (strcmp(module, prevmodule) != 0) {
	    prevmodule = xmalloc(strlen(module)+1);
	    strcpy(prevmodule, module);
	}
	entry->module_name = prevmodule;
    }
    fclose(pcimap_file);
}

static int pci_match_vendor (unsigned int vendor, unsigned int device,
			 unsigned int subvendor, unsigned int subdevice,
			 unsigned int pci_class)
{
    int i;
    int retval;
    unsigned int class_temp;

    dbg ("vendor = %x, device = %x, subvendor = %x, subdevice = %x",
	    vendor, device, subvendor, subdevice);

    for (i = 0; pci_module_map[i].module_name != NULL; ++i) {
	dbg ("looking at %s", pci_module_map[i].module_name);
	if ((pci_module_map[i].vendor != PCI_ANY) &&
		(pci_module_map[i].vendor != vendor)) {
	    dbg ("vendor check failed %x != %x",
		    pci_module_map[i].vendor, vendor);
	    continue;
	}
	if ((pci_module_map[i].device != PCI_ANY) &&
		(pci_module_map[i].device != device)) {
	    dbg ("device check failed %x != %x",
		    pci_module_map[i].device, device);
	    continue;
	}
	if ((pci_module_map[i].subvendor != PCI_ANY) &&
		(pci_module_map[i].subvendor != subvendor)) {
	    dbg ("subvendor check failed %x != %x",
		    pci_module_map[i].subvendor, subvendor);
	    continue;
	}
	if ((pci_module_map[i].subdevice != PCI_ANY) &&
		(pci_module_map[i].subdevice != subdevice)) {
	    dbg ("subdevice check failed %x != %x",
		    pci_module_map[i].subdevice, subdevice);
	    continue;
	}

	/* check that the class matches */
	class_temp = (pci_module_map[i].class ^ pci_class) & pci_module_map[i].class_mask;
	if (class_temp != 0) {
	    dbg ("class mask check failed %x != %x",
		    pci_module_map[i].class, class_temp);
	    continue;
	}

	/* found one! */
	dbg ("loading %s", pci_module_map[i].module_name);
	retval = load_module (pci_module_map[i].module_name);
	if (retval)
	    return retval;
    }

    return 0;
}

static int pci_add (void)
{
    char *class_env;
    char *id_env;
    char *subsys_env;
    int error;
    unsigned int vendor;
    unsigned int device;
    unsigned int subvendor;
    unsigned int subdevice;
    unsigned int class;

    pci_read_pcimap();

    id_env = getenv("PCI_ID");
    subsys_env = getenv("PCI_SUBSYS_ID");
    class_env = getenv("PCI_CLASS");
    if ((id_env == NULL) ||
	    (subsys_env == NULL) ||
	    (class_env == NULL)) {
	dbg ("missing an environment variable, aborting.");
	return 1;
    }

    error = split_2values (id_env, 16, &vendor, &device);
    if (error)
	return error;
    error = split_2values (subsys_env, 16, &subvendor, &subdevice);
    if (error)
	return error;
    class = strtoul (class_env, NULL, 16);

    error = pci_match_vendor (vendor, device, subvendor, subdevice, class);

    return error;
}


static int pci_remove (void)
{
    /* right now we don't do anything here :) */
    return 0;
}

static struct subsystem pci_subsystem[] = {
    { ADD_STRING,	pci_add },
    { REMOVE_STRING,	pci_remove },
    { NULL,		NULL }
};

int pci_handler (void)
{
    char * action;

    action = getenv ("ACTION");
    dbg ("action = %s", action);
    if (action == NULL) {
	dbg ("missing ACTION environment variable, aborting.");
	return 1;
    }
    return call_subsystem (action, pci_subsystem);
}


/* The usb functions */
#define USB_DEVICE_ID_MATCH_VENDOR		0x0001
#define USB_DEVICE_ID_MATCH_PRODUCT		0x0002
#define USB_DEVICE_ID_MATCH_DEV_LO		0x0004
#define USB_DEVICE_ID_MATCH_DEV_HI		0x0008
#define USB_DEVICE_ID_MATCH_DEV_CLASS		0x0010
#define USB_DEVICE_ID_MATCH_DEV_SUBCLASS	0x0020
#define USB_DEVICE_ID_MATCH_DEV_PROTOCOL	0x0040
#define USB_DEVICE_ID_MATCH_INT_CLASS		0x0080
#define USB_DEVICE_ID_MATCH_INT_SUBCLASS	0x0100
#define USB_DEVICE_ID_MATCH_INT_PROTOCOL	0x0200
#define USB_DEVICE_ID_MATCH_DEVICE		(USB_DEVICE_ID_MATCH_VENDOR | USB_DEVICE_ID_MATCH_PRODUCT)
#define USB_DEVICE_ID_MATCH_DEV_RANGE		(USB_DEVICE_ID_MATCH_DEV_LO | USB_DEVICE_ID_MATCH_DEV_HI)
#define USB_DEVICE_ID_MATCH_DEVICE_AND_VERSION	(USB_DEVICE_ID_MATCH_DEVICE | USB_DEVICE_ID_MATCH_DEV_RANGE)
#define USB_DEVICE_ID_MATCH_DEV_INFO		(USB_DEVICE_ID_MATCH_DEV_CLASS | \
						USB_DEVICE_ID_MATCH_DEV_SUBCLASS | \
						USB_DEVICE_ID_MATCH_DEV_PROTOCOL)
#define USB_DEVICE_ID_MATCH_INT_INFO		(USB_DEVICE_ID_MATCH_INT_CLASS | \
						USB_DEVICE_ID_MATCH_INT_SUBCLASS | \
						USB_DEVICE_ID_MATCH_INT_PROTOCOL)

struct usb_module_map {
	const char * module_name;
	unsigned int match_flags;
	unsigned int idVendor;
	unsigned int idProduct;
	unsigned int bcdDevice_lo;
	unsigned int bcdDevice_hi;
	unsigned int bDeviceClass;
	unsigned int bDeviceSubClass;
	unsigned int bDeviceProtocol;
	unsigned int bInterfaceClass;
	unsigned int bInterfaceSubClass;
	unsigned int bInterfaceProtocol;
} __attribute__ ((packed));
static struct usb_module_map *usb_module_map;

void usb_read_usbmap(void)
{
    FILE *usbmap_file;
    struct utsname utsname;
    struct usb_module_map *entry = usb_module_map;
    char *prevmodule = "";
    char filename[MAXPATHLEN];
    char line[LINELENGTH];
    char module[LINELENGTH];
    int count, mapsize;

    if (uname(&utsname) < 0) {
	perror("uname");
	exit(1);
    }
    sprintf(filename, "/lib/modules/%s/modules.usbmap", utsname.release);
    if ((usbmap_file = fopen(filename, "r")) == NULL) {
	bb_perror_msg_and_die("Could not open %s", filename);
    }

    count = mapsize = 0;
    while(fgets(line, LINELENGTH, usbmap_file) != NULL) {

	if (line[0] == '#')
	    continue;

	count++;
	if (count >= mapsize) {
	    mapsize+=100;
	    usb_module_map = xrealloc(usb_module_map, (sizeof(struct usb_module_map) * mapsize));
	    entry = usb_module_map + count - 1;
	    memset(entry, 0, (sizeof(struct usb_module_map) * 100));
	}
	entry = usb_module_map + count - 1;

	if (sscanf(line, "%s 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x 0x%x",
		    module,
		    &entry->match_flags, &entry->idVendor,
		    &entry->idProduct, &entry->bcdDevice_lo,
		    &entry->bcdDevice_hi, &entry->bDeviceClass,
		    &entry->bDeviceSubClass, &entry->bDeviceProtocol,
		    &entry->bInterfaceClass, &entry->bInterfaceSubClass,
		    &entry->bInterfaceProtocol) != 12) {
	    bb_error_msg("modules.usbmap unparsable line: %s.\n", line);
	    free(entry);
	    continue;
	}

	/* Optimize memory allocation a bit, in case someday we
	   have Linux systems with ~100,000 modules.  It also
	   allows us to just compare pointers to avoid trying
	   to load a module twice. */
	if (strcmp(module, prevmodule) != 0) {
	    prevmodule = xmalloc(strlen(module)+1);
	    strcpy(prevmodule, module);
	}
	entry->module_name = prevmodule;
    }
    fclose(usbmap_file);
}

static int usb_match_vendor_product (unsigned int vendor, unsigned int product, unsigned int bcdDevice)
{
    int i;
    int retval;

    dbg ("vendor = %x, product = %x, bcdDevice = %x", vendor, product, bcdDevice);

    for (i = 0; usb_module_map[i].module_name != NULL; ++i) {
	dbg ("looking at %s, match_flags = %x", usb_module_map[i].module_name, usb_module_map[i].match_flags);
	if (usb_module_map[i].match_flags & (USB_DEVICE_ID_MATCH_DEVICE | USB_DEVICE_ID_MATCH_DEV_RANGE)) {
	    if ((usb_module_map[i].match_flags & USB_DEVICE_ID_MATCH_VENDOR) &&
		    (usb_module_map[i].idVendor != vendor)) {
		dbg ("vendor check failed %x != %x", usb_module_map[i].idVendor, vendor);
		continue;
	    }
	    if ((usb_module_map[i].match_flags & USB_DEVICE_ID_MATCH_PRODUCT) &&
		    (usb_module_map[i].idProduct != product)) {
		dbg ("product check failed %x != %x", usb_module_map[i].idProduct, product);
		continue;
	    }
	    if ((usb_module_map[i].match_flags & USB_DEVICE_ID_MATCH_DEV_LO) &&
		    (usb_module_map[i].bcdDevice_lo < bcdDevice)) {
		dbg ("bcdDevice_lo check failed %x > %x", usb_module_map[i].bcdDevice_lo, bcdDevice);
		continue;
	    }
	    if ((usb_module_map[i].match_flags & USB_DEVICE_ID_MATCH_DEV_HI) &&
		    (usb_module_map[i].bcdDevice_hi >= bcdDevice)) {
		dbg ("bcdDevice_hi check failed %x <= %x", usb_module_map[i].bcdDevice_hi, bcdDevice);
		continue;
	    }
	    /* found one! */
	    dbg ("loading %s", usb_module_map[i].module_name);
	    retval = load_module (usb_module_map[i].module_name);
	    if (retval)
		return retval;
	}
    }

    return 0;
}

static int usb_match_device_class (unsigned int class, unsigned int subclass, unsigned int protocol)
{
    int i;
    int retval;

    dbg ("class = %x, subclass = %x, protocol = %x", class, subclass, protocol);

    for (i = 0; usb_module_map[i].module_name != NULL; ++i) {
	dbg ("looking at %s, match_flags = %x", usb_module_map[i].module_name, usb_module_map[i].match_flags);
	if (usb_module_map[i].match_flags & USB_DEVICE_ID_MATCH_DEV_INFO) {
	    if ((usb_module_map[i].match_flags & USB_DEVICE_ID_MATCH_DEV_CLASS) &&
		    (usb_module_map[i].bDeviceClass != class)) {
		dbg ("class check failed %x != %x", usb_module_map[i].bDeviceClass, class);
		continue;
	    }
	    if ((usb_module_map[i].match_flags & USB_DEVICE_ID_MATCH_DEV_SUBCLASS) &&
		    (usb_module_map[i].bDeviceSubClass != subclass)) {
		dbg ("subclass check failed %x != %x", usb_module_map[i].bDeviceSubClass, subclass);
		continue;
	    }
	    if ((usb_module_map[i].match_flags & USB_DEVICE_ID_MATCH_DEV_PROTOCOL) &&
		    (usb_module_map[i].bDeviceProtocol != protocol)) {
		dbg ("protocol check failed %x != %x", usb_module_map[i].bDeviceProtocol, protocol);
		continue;
	    }
	    /* found one! */
	    dbg ("loading %s", usb_module_map[i].module_name);
	    retval = load_module (usb_module_map[i].module_name);
	    if (retval)
		return retval;
	}
    }

    return 0;
}


static int usb_match_interface_class (unsigned int class, unsigned int subclass, unsigned int protocol)
{
    int i;
    int retval;

    dbg ("class = %x, subclass = %x, protocol = %x", class, subclass, protocol);

    for (i = 0; usb_module_map[i].module_name != NULL; ++i) {
	dbg ("looking at %s, match_flags = %x", usb_module_map[i].module_name, usb_module_map[i].match_flags);
	if (usb_module_map[i].match_flags & USB_DEVICE_ID_MATCH_INT_INFO) {
	    if ((usb_module_map[i].match_flags & USB_DEVICE_ID_MATCH_INT_CLASS) &&
		    (usb_module_map[i].bInterfaceClass != class)) {
		dbg ("class check failed %x != %x", usb_module_map[i].bInterfaceClass, class);
		continue;
	    }
	    if ((usb_module_map[i].match_flags & USB_DEVICE_ID_MATCH_INT_SUBCLASS) &&
		    (usb_module_map[i].bInterfaceSubClass != subclass)) {
		dbg ("subclass check failed %x != %x", usb_module_map[i].bInterfaceSubClass, subclass);
		continue;
	    }
	    if ((usb_module_map[i].match_flags & USB_DEVICE_ID_MATCH_INT_PROTOCOL) &&
		    (usb_module_map[i].bInterfaceProtocol != protocol)) {
		dbg ("protocol check failed %x != %x", usb_module_map[i].bInterfaceProtocol, protocol);
		continue;
	    }
	    /* found one! */
	    dbg ("loading %s", usb_module_map[i].module_name);
	    retval = load_module (usb_module_map[i].module_name);
	    if (retval)
		return retval;
	}
    }

    return 0;
}


static int usb_add (void)
{
    char *product_env;
    char *type_env;
    char *interface_env;
    int error;
    unsigned int idVendor;
    unsigned int idProduct;
    unsigned int bcdDevice;
    unsigned int device_class;
    unsigned int device_subclass;
    unsigned int device_protocol;
    unsigned int interface_class = 0;
    unsigned int interface_subclass = 0;
    unsigned int interface_protocol = 0;

    usb_read_usbmap();

    product_env = getenv ("PRODUCT");
    type_env = getenv ("TYPE");
    if ((product_env == NULL) ||
	    (type_env == NULL)) {
	dbg ("missing an environment variable, aborting.");
	return 1;
    }
    error = split_3values (product_env, 16, &idVendor, &idProduct, &bcdDevice);
    if (error)
	return error;

    error = usb_match_vendor_product (idVendor, idProduct, bcdDevice);
    if (error)
	return error;

    error = split_3values (type_env, 10, &device_class,
	    &device_subclass, &device_protocol);
    if (error)
	return error;
    error = usb_match_device_class (device_class, device_subclass, device_protocol);
    if (error)
	return error;

    /* we need to look at the interface too */
    interface_env = getenv ("INTERFACE");
    if (interface_env == NULL) {
	dbg ("interface is null, we don't know what to do here.");
	return 1;
    }
    error = split_3values (interface_env, 10, &interface_class,
	    &interface_subclass, &interface_protocol);
    if (error)
	return error;
    error = usb_match_interface_class (interface_class, interface_subclass, interface_protocol);
    return error;
}


static int usb_remove (void)
{
    /* right now we don't do anything here :) */
    return 0;
}


static struct subsystem usb_subsystem[] = {
    { ADD_STRING,	usb_add },
    { REMOVE_STRING,	usb_remove },
    { NULL,		NULL }
};


int usb_handler (void)
{
    char * action;

    action = getenv ("ACTION");
    dbg ("action = %s", action);
    if (action == NULL) {
	dbg ("missing ACTION environment variable, aborting.");
	return 1;
    }
    return call_subsystem (action, usb_subsystem);
}


/* Utility functions needed by some of the subsystems. */

/*
 * split_2values
 *
 * takes a string of format "xxxx:yyyy" and figures out the
 * values for xxxx and yyyy
 */
int split_2values (const char *string, int base, unsigned int *value1, unsigned int *value2)
{
    char buffer[200];
    char *temp1;
    const char *temp2;

    dbg("string = %s", string);

    if (string == NULL)
	return -1;
    if (strnlen (string, sizeof(buffer)) >= sizeof(buffer))
	return -1;

    /* pick up the first number */
    temp1 = &buffer[0];
    temp2 = string;
    while (1) {
	if (*temp2 == 0x00)
	    break;
	if (*temp2 == ':')
	    break;
	*temp1 = *temp2;
	++temp1;
	++temp2;
    }
    *temp1 = 0x00;
    *value1 = strtoul (buffer, NULL, base);
    dbg ("buffer = %s", &buffer[0]);
    dbg ("value1 = %d", *value1);

    if (*temp2 == 0x00) {
	/* string is ended, not good */
	return -1;
    }

    /* get the second number */
    temp1 = &buffer[0];
    ++temp2;
    while (1) {
	if (*temp2 == 0x00)
	    break;
	*temp1 = *temp2;
	++temp1;
	++temp2;
    }
    *temp1 = 0x00;
    *value2 = strtoul (buffer, NULL, base);
    dbg ("buffer = %s", &buffer[0]);
    dbg ("value2 = %d", *value2);

    return 0;
}


/*
 * split_3values
 *
 * takes a string of format "xxxx/yyyy/zzzz" and figures out the
 * values for xxxx, yyyy, and zzzz
 */
int split_3values (const char *string, int base, unsigned int * value1,
	unsigned int * value2, unsigned int * value3)
{
    char buffer[200];
    char *temp1;
    const char *temp2;

    dbg("string = %s", string);

    if (string == NULL)
	return -1;
    if (strnlen (string, sizeof(buffer)) >= sizeof(buffer))
	return -1;

    /* pick up the first number */
    temp1 = &buffer[0];
    temp2 = string;
    while (1) {
	if (*temp2 == 0x00)
	    break;
	if (*temp2 == '/')
	    break;
	*temp1 = *temp2;
	++temp1;
	++temp2;
    }
    *temp1 = 0x00;
    *value1 = strtoul (buffer, NULL, base);
    dbg ("buffer = %s", &buffer[0]);
    dbg ("value1 = %d", *value1);

    if (*temp2 == 0x00) {
	/* string is ended, not good */
	return -1;
    }

    /* get the second number */
    temp1 = &buffer[0];
    ++temp2;
    while (1) {
	if (*temp2 == 0x00)
	    break;
	if (*temp2 == '/')
	    break;
	*temp1 = *temp2;
	++temp1;
	++temp2;
    }
    *temp1 = 0x00;
    *value2 = strtoul (buffer, NULL, base);
    dbg ("buffer = %s", &buffer[0]);
    dbg ("value2 = %d", *value2);

    if (*temp2 == 0x00) {
	/* string is ended, not good */
	return -1;
    }

    /* get the third number */
    temp1 = &buffer[0];
    ++temp2;
    while (1) {
	if (*temp2 == 0x00)
	    break;
	*temp1 = *temp2;
	++temp1;
	++temp2;
    }
    *temp1 = 0x00;
    *value3 = strtoul (buffer, NULL, base);
    dbg ("buffer = %s", &buffer[0]);
    dbg ("value3 = %d", *value3);

    return 0;
}


int call_subsystem (const char *string, struct subsystem *subsystem)
{
    int i;

    for (i = 0; subsystem[i].name != NULL; ++i) {
	if (strncmp (string, subsystem[i].name, strlen(string)) == 0)
	    return subsystem[i].handler();
    }
    return 1;
}


int load_module (const char *module_name)
{
    char *argv[3];

    argv[0] = "/sbin/modprobe";
    argv[1] = (char *)module_name;
    argv[2] = NULL;
    dbg ("loading module %s", module_name);
    switch (fork()) {
	case 0:
	    /* we are the child, so lets run the program */
	    execv ("/sbin/modprobe", argv);
	    exit(0);
	    break;
	case (-1):
	    dbg ("fork failed.");
	    break;
	default:
	    break;
    }
    return 0;
}

